<?php

namespace Verifone\Hosted\Controller\Standard;

use Magento\Framework\App\CsrfAwareActionInterface;
use Magento\Framework\App\Request\InvalidRequestException;
use Magento\Framework\App\RequestInterface;
use Verifone\Hosted\Controller\Checkout;
use Verifone\Hosted\Helper\Verifone;
use VerifoneEcomAPI\ApiWrapper\Http\SimpleCurl;

class Webhook extends Checkout implements CsrfAwareActionInterface
{
    protected $_simpleCurl;
    protected $_orderRepo;
    protected $_transactionBuilder;
    protected $response;
    protected $request;

    public function __construct(
        \Magento\Framework\App\Action\Context                           $context,
        \Magento\Customer\Model\Session                                 $customerSession,
        \Magento\Checkout\Model\Session                                 $checkoutSession,
        \Magento\Quote\Api\CartRepositoryInterface                      $quoteRepository,
        \Magento\Sales\Model\OrderFactory                               $orderFactory,
        \Psr\Log\LoggerInterface                                        $logger,
        \Verifone\Hosted\Model\Verifone                                 $paymentMethod,
        \Verifone\Hosted\Helper\Verifone                                $checkoutHelper,
        \Magento\Quote\Api\CartManagementInterface                      $cartManagement,
        \Magento\Framework\Controller\Result\JsonFactory                $resultJsonFactory,
        \Magento\Sales\Api\OrderRepositoryInterface                     $orderRepo,
        \Magento\Sales\Model\Order\Payment\Transaction\BuilderInterface $transactionBuilder,
        \Magento\Framework\App\Request\Http                             $request,
        \Magento\Framework\App\Response\Http                            $response
    ) {
        parent::__construct($context, $customerSession, $checkoutSession, $quoteRepository, $orderFactory, $logger, $paymentMethod, $checkoutHelper, $cartManagement, $resultJsonFactory);

        $this->_simpleCurl = new SimpleCurl();
        $this->_orderRepo = $orderRepo;
        $this->_transactionBuilder = $transactionBuilder;
        $this->request = $request;
        $this->response = $response;

    }

    public function execute()
    {
        $input = $this->request->getContent();
        if (!$input) {
            $this->api_response_error('Unable to process data');
        }

        $data = json_decode($input, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            $this->_logger->error(__CLASS__ . '@' . __FUNCTION__ . ' line ' . __LINE__ . ': Unable to decode response from API');
            $this->api_response_error('Unable to process data');
        }

        $paymentMethod = $this->getPaymentMethod();

        if ($this->stringStartsWith($data['eventType'], 'Checkout')) {
            try {
                $checkout = $this->getCheckoutHelper()->getCheckout($paymentMethod, $data['recordId']);
                $data['recordId'] = $checkout['transaction_id'];
            } catch (\Exception $exception) {
                $this->_logger->error(__CLASS__ . '@' . __FUNCTION__ . ' line ' . __LINE__ . ': There was a request for an order: ' . $exception->getMessage());
                $this->api_response_error('Unable to process data');
            }
        }

        $response = $this->getCheckoutHelper()->getTransaction($paymentMethod, $data['recordId']);
        $transactionId = $response['id'];
        $amountPaid = $this->_checkoutHelper->convertAmountToVerifone($response['amount'], $response['currency_code']);

        if (!empty($response['merchant_reference'])) {
            $quote = $this->_quoteRepository->get($response['merchant_reference']);
        } elseif (!empty($response['id'])) {
            $quote = $this->_quoteRepository->get($response['id']);
        } else {
            $this->_logger->error(__CLASS__ . '@' . __FUNCTION__ . ' line ' . __LINE__ . ': API did not sent the ID of the quote');
            $this->api_response_error('Unable to process data');
        }

        $order = $this->getCheckoutHelper()->getOrderByIncrementId($quote->getReservedOrderId());

        switch ($response['status']) {
            case Verifone::VERIFONE_TRANSACTION_CAPTURE_AUTHORISED:
            case Verifone::VERIFONE_TRANSACTION_CAPTURE_SETTLED:
            case Verifone::VERIFONE_TRANSACTION_AUTHORISED:
            case Verifone::VERIFONE_TRANSACTION_SALE_AUTHORISED:
            case Verifone::VERIFONE_TRANSACTION_SALE_SETTLED:
            case Verifone::VERIFONE_TRANSACTION_AUTHORISATION_AUTHORISED:
            case Verifone::VERIFONE_TRANSACTION_CARD_VERIFICATION_AUTHORISED:
                if (!$order) {
                    $code = $paymentMethod->getCode();
                    $quote->setPaymentMethod($code);
                    $quote->getPayment()->importData(['method' => $code]);
                    $this->_quoteRepository->save($quote);

                    try {
                        $this->_cartManagement->placeOrder(
                            $quote->getId(),
                            $quote->getPayment()
                        );
                        $order = $this->getOrder();
                    } catch (\Exception $e) {
                        $this->_logger->error($e->getMessage());
                        $this->_logger->error($e->getTraceAsString());
                        if (isset($this->messageManager)) {
                            $this->messageManager->addExceptionMessage($e, __('We can\'t place the order.'));
                        }
                    }
                }
                $this->getCheckoutSession()
                    ->setLastSuccessQuoteId($quote->getId())
                    ->setLastQuoteId($quote->getId())
                    ->setLastRealOrderId($order->getIncrementId())
                    ->setLastOrderId($order->getId())
                    ->setLastOrderStatus($order->getStatus());

                /** Because with a refund, we can receive a transaction event but also a checkout event having transaction_id the original successful transaction, check that we don't already have a refunded amount before set status to complete.
                 */
                if (!$this->_checkoutHelper->checkVerifoneTransactionIdInOrderLogHistory($transactionId, $order) && !($order->getTotalRefunded() ?? 0)) {
                    $payment = $order->getPayment();
                    $paymentMethod = $payment->getMethodInstance();
                    $this->_checkoutHelper->setUpVerifoneSettings($paymentMethod);

                    $order->setExtOrderId($response['id']);
                    $order->save();

                    $manualCapture = $this->getCheckoutHelper()->checkManualCapture();
                    $response['transaction_id'] = $transactionId;
                    $paymentMethod->postProcessing($order, ['transaction_id' => $transactionId], $manualCapture, $response['payment_product_type'], (int) $amountPaid);
                    $order->save();
                }
                break;

            case Verifone::VERIFONE_TRANSACTION_PENDING:
                if (!$order) {
                    break;
                }

                $order->setExtOrderId($response['id'])
                    ->addCommentToStatusHistory(__('The payment is not done yet (transaction is in PRE/FINAL AUTH status), use the CAPTURE button to fully/partial capture the amount.
                    Attempting to refund the order in part or in full, before capturing, will void the authorization and cancel the payment.'))
                    ->save();
                break;

            default:
                if (!$order) {
                    break;
                }

                $order->addCommentToStatusHistory(__('Transaction event logged from Verifone: ') . $data['eventType'] . ' Response status: ' . $response['status'])
                    ->save();
        }

        return $this->api_response_ok();
    }

    public function api_response_error($message = null)
    {
        $this->api_response($message, 500);
    }

    public function api_response($message = null, $statusCode = 200)
    {
        $this->response->setStatusCode($statusCode);
        return;
    }

    public function createCsrfValidationException(RequestInterface $request): ?InvalidRequestException
    {
        return null;
    }

    public function validateForCsrf(RequestInterface $request): ?bool
    {
        return true;
    }

    public function api_response_ok($message = null)
    {
        $this->api_response($message, 200);
    }

    public function stringStartsWith($string, $needle)
    {
        $len = strlen($needle);

        return (substr($string, 0, $len) === $needle);
    }
}
