<?php

namespace Overdose\InStorePickup\ViewModel;

use Overdose\InStorePickup\Model\Service\SourcesService;
use Overdose\InStorePickup\Model\Service\SourceItemService;
use Magento\Framework\View\Element\Block\ArgumentInterface;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Store\Model\StoreFactory;
use BrittainWynyard\CatalogSuperStyle\Block\Swatch;
use Magento\Catalog\Helper\Product;
use Magento\Catalog\Model\ProductRepository;
use Magento\Framework\Json\EncoderInterface;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Overdose\InStorePickup\Helper\Data as Helper;
use Psr\Log\LoggerInterface;
use Exception;

class StockInfo implements ArgumentInterface
{

    /**
     * @var StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var
     */
    protected $storeFactory;

    /**
     * @var array
     */
    private $list = array();

    /**
     * @var array
     */
    private $listAdditional = array();

    /**
     * @var Swatch
     */
    private $productSwatch;

    /**
     * @var Product
     */
    protected $catalogProduct;

    /**
     * @var ProductRepository
     */
    protected $productRepository;

    /**
     * @var SearchCriteriaBuilder
     */
    protected $searchCriteriaBuilder;
    protected $productAttributes = [];
    protected $saleableSku = [];
    protected $allSimpleSku = [];
    protected $msiArray = [];
    protected $msiStock = [];
    protected $colorAttr = [];

    /**
     * @var Json
     */
    protected $jsonEncoder;

    /**
     * @var
     */
    protected $msiStoreSize;

    /**
     * @var SourcesService
     */
    private $sourcesService;

    /**
     * @var SourceItemService
     */
    private $sourceItemService;

    /**
     * @var LoggerInterface
     */
    private $logger;

    private $storeAllSourceList = [];

    /**
     * @var Helper
     */
    protected $stockHelper;


    /**
     * @param StoreManagerInterface $storeManager
     * @param StoreFactory $storeFactory
     * @param Swatch $productSwatch
     * @param Product $catalogProduct
     * @param ProductRepository $productRepository
     * @param SourceItemService $sourceItemService
     * @param Json $jsonEncoder
     * @param SourcesService $sourcesService
     * @param Helper $stockHelper
     * @param LoggerInterface $logger
     */
    public function __construct(

        StoreManagerInterface $storeManager,
        StoreFactory          $storeFactory,
        Swatch                $productSwatch,
        Product               $catalogProduct,
        ProductRepository     $productRepository,
        SourceItemService     $sourceItemService,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        Json                  $jsonEncoder,
        SourcesService        $sourcesService,
        Helper                $stockHelper,
        LoggerInterface       $logger
    ) {
        $this->storeManager = $storeManager;
        $this->sourcesService = $sourcesService;
        $this->sourceItemService = $sourceItemService;
        $this->catalogProduct = $catalogProduct;
        $this->productRepository = $productRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->stockHelper = $stockHelper;
        $this->productSwatch = $productSwatch;
        $this->jsonEncoder = $jsonEncoder;
        $this->logger = $logger;
    }

    /**
     *  get related products from swatch
     */
    public function getRelatedProducts()
    {
        return $this->productSwatch->getRelatedProducts();
    }

    /**
     * get current ProductId
     */
    public function getCurrentProductId()
    {
        $currentProduct = $this->productSwatch->getProduct();
        return $currentProduct->getId();
    }

    /**
     * Set product for AJAX requests
     */
    public function setProduct($product)
    {
        $this->productSwatch->setProduct($product);
        return $this;
    }


    /**
     * get initial color attribute using current product
     */
    public function getInitColorAttributes()
    {
        $relatedProducts = $this->getRelatedProducts();
        $currentProductId = $this->getCurrentProductId();
        
        foreach ($relatedProducts as $product) {
            $colorArray = array();
            $colourAttr = 'Undefined';
            if ($product->hasData("color")) {
                $colourAttr = $product->getAttributeText('color');
            }
            if ($product->hasData("mkt_colour")) {
                $_mkcolourAttr = $product->getResource()->getAttribute('mkt_colour')->getFrontend()->getValue($product);
                if ($_mkcolourAttr) {
                    $colourAttr = $_mkcolourAttr;
                }
            }

            $pid = $product->getId();
            $colorArray['pid'] = $pid;
            $colorArray['option_title'] = $colourAttr;
            $colorArray['option_value'] = __('Colour: ') . $colourAttr;
            $colorArray['is_current'] = ($pid == $currentProductId); // Mark current product
            $this->colorAttr[] = $colorArray;

        }
        return $this->jsonEncoder->serialize($this->colorAttr);
    }


    /**
     * get initial msi stock
     */
    public function getInitMsiStock()
    {
        return $this->jsonEncoder->serialize($this->msiStock);
    }

    /**
     * Init store size when loading product page
     */
    public function getInitMsiStoreSize()
    {
        return $this->jsonEncoder->serialize($this->msiStoreSize);
    }

    /**
     * Get store identifier
     *
     * @return  int
     */
    public function getStoreId()
    {
        return $this->storeManager->getStore()->getId();
    }

    /**
     * Is extension active
     */
    public function isActive()
    {
        return $this->stockHelper->isActive($this->getStoreId());
    }

    /**
     * Is extension active
     */
    public function showOutofStock()
    {
        return $this->stockHelper->showOutofstock($this->getStoreId());
    }

    public function getSizeAvailableContent()
    {
        return $this->jsonEncoder->serialize($this->stockHelper->getModelContent($this->getStoreId()));
    }

    /**
     * get Product MSI
     */
    public function getMsiList()
    {
        $msiList = array();
        //only for  product has color
        if (!empty($this->colorAttr)) {
            if (empty($this->storeAllSourceList)) {
                $this->getStoreAllSourcesList();
            }
            
            // Extract all product IDs for batch loading
            $productIds = array_column($this->colorAttr, 'pid');
            
            // Load all products in a single query using getList
            $searchCriteria = $this->searchCriteriaBuilder
                ->addFilter('entity_id', $productIds, 'in')
                ->create();
            $products = $this->productRepository->getList($searchCriteria);
            $productMap = [];
            
            // Create a map of product ID to product object for quick lookup
            foreach ($products->getItems() as $product) {
                $productMap[$product->getId()] = $product;
            }
            
            // Process each product using the pre-loaded data
            foreach ($this->colorAttr as $colourAttr) {
                $pid = $colourAttr['pid'];
                if (isset($productMap[$pid])) {
                    $product = $productMap[$pid];
                    $this->generateMsiData($product);
                    $msiList[$pid] = $this->msiArray;
                    $this->msiArray = [];
                }
            }
            return $this->jsonEncoder->serialize($msiList);
        } else {
            return $this->jsonEncoder->serialize($msiList);
        }
    }

    /**
     * get saleable simpe products in configurable product
     */
    public function getAllowProducts($product)
    {
        $skipSaleableCheck = $this->catalogProduct->getSkipSaleableCheck();
        $products = $skipSaleableCheck ?
            $product->getTypeInstance()->getUsedProducts($product, null) :
            $product->getTypeInstance()->getSalableUsedProducts($product, null);
        return $products;
    }

    /**
     * Check whether exist width attribute
     */
    public function getAttributeOption($attributeData)
    {
        $options = array();
        $attributCodes = array();
        foreach ($attributeData as $attr) {
            foreach ($attr as $p) {
                if (!in_array($p['attribute_code'], $attributCodes)) {
                    $attributCodes[] = $p['attribute_code'];
                }
                $options[$p['sku']] = $p['option_title'];
            }
        }
        return $options;
    }


    /**
     * Generate Msi Data for simple and configurable product
     */
    public function generateMsiData($product)
    {
        if ($product->getTypeId() == 'configurable') {
            $product->setUseForPopupStockAvibility(true);
            $attributeData = $product->getTypeInstance()->getConfigurableOptions($product);
            $allSimpleProduct = $this->getAllSimpleProduct($product, $attributeData);
            $widthArray = $this->getMsiSkuDataIncludeOutStock($attributeData);
            $this->getMsiInventoryData($allSimpleProduct, $widthArray); //generate msi inventory data

            //set Init width array to the current product color from generated msi inventory data
            if ($product->getId() == $this->getCurrentProductId()) {
                $this->generateInitMsiStock($this->getCurrentProductId());
            }
        } else if ($product->getTypeId() == 'simple') {
            $this->getSimpleMsiInventoryData($product);
            $this->msiStock = $this->msiArray[0]['msi_lists'];
            $this->msiStoreSize = $this->msiArray[0]['store_size'];
        }
    }

    public function getSimpleMsiInventoryData($product)
    {
        $tempSourceList = array();  // save list of sku stock  in msi
        $sourceCodes = array();  // save configurable existing sourceCode, as some product might only assign to one source, the other might assign to multiple source

        $msi = array();
        $msi['product_id'] = $product->getId();

        $msi['sku'] = $product->getSku();
        $msiArray[] = $msi;
        $sku = $product->getSku();
        if (!empty($sku)) {
            $tempSourceList[$sku] = $this->getSourceData($sku);
            //Get MSI source_code for saleable associated simple products
            foreach ($tempSourceList as $sku => $sources) {
                foreach ($sources as $source) {
                    if (!in_array($source->getData('source_code'), $sourceCodes)) {
                        $sourceCodes[] = $source->getData('source_code');
                    }
                }
            }
            if (count($sourceCodes) > 1 && in_array("default", $sourceCodes)) { //set default store as first element
                // Search
                if (($key = array_search('default', $sourceCodes)) !== false) {
                    unset($sourceCodes[$key]);
                    array_unshift($sourceCodes, 'default');
                }
            }

            //Form the MSI sources stock information for each saleable simple products. Such as default or birkenstock_newmarket
            foreach ($sourceCodes as $sourceCode) {
                $sourceData['source_code'] = $sourceCode;

                if ($this->storeAllSourceList[$sourceCode]) {
                    $sourceData['source_info'] = $this->storeAllSourceList[$sourceCode];
                }
                $sourceData["sku_stock"] = array();
                $sourceData["store_size"] = array();
                $sourceData["stock_availability"] = array();  // 0 means sold out,  1 means low stock, 2 means In stock

                $stockAvailability = array();

                foreach ($tempSourceList as $sku => $sources) {
                    $skuExistSourceCode = array();
                    $skuSource = array();
                    $size = array();
                    $color = array();
                    $stockAvailability = array();
                    foreach ($sources as $source) {
                        //get sku assigned MSI source
                        if (!in_array($source->getData('source_code'), $skuExistSourceCode)) {
                            $skuExistSourceCode[] = $source->getData('source_code');
                        }
                        //if sku assigned to the sourceCode, just fetch the stock value for the sku
                        if (in_array($sourceCode, $skuExistSourceCode)) {
                            if ($source->getData('source_code') == $sourceCode) { //get sku source qty
                                $skuSource['sku'] = $sku;
                                $skuSource['qty'] = $source->getData('quantity');
                                if ($skuSource['qty'] <= 0) {
                                    $stockAvailability['stock_availability'] = "0";
                                } else if ($skuSource['qty'] == 1) {
                                    $stockAvailability['stock_availability'] = "1";
                                } else {
                                    $stockAvailability['stock_availability'] = "2";
                                }

                                if ($product->hasData("size")) {
                                    $skuSource['size'] = $product->getAttributeText("size");
                                    $size['size'] = $product->getAttributeText("size");
                                }

                            }
                        } else { // assign it to 0 if sku is not ssigned to the sourceCode
                            $skuSource['sku'] = $sku;
                            $stockAvailability['stock_availability'] = 0;
                            $skuSource['qty'] = $source->getData('quantity');

                            if ($product->hasData("size")) {
                                $skuSource['size'] = $product->getAttributeText("size");
                                $size['size'] = $product->getAttributeText("size");
                            }

                        }
                    }
                    $sourceData["sku_stock"][] = $skuSource;
                    $sourceData["store_size"][] = $size;
                    $sourceData["stock_availability"][] = $stockAvailability;
                }
                $this->msiArray[0]['store_size'] = $sourceData["store_size"];//// used to show size in popup directly
                $this->msiArray[0]['msi_lists'][] = $sourceData;
            }
        }

    }

    /**
     * Get MSI SKu inventory
     */
    public function getMsiInventoryData($productList, $widthArray)
    {

        $sourceCodes = array();  // save configurable existing sourceCode, as some product might only assign to one source, the other might assign to multiple source
        for ($i = 0; $i < count($this->msiArray); $i++) {
            $tempSourceList = array();  // save list of sku stock  in msi
            $sku = $this->msiArray[$i]['sku'];
            if (!empty($sku)) {
                if (strpos($sku, ",")) {
                    $skus = explode(",", $sku);
                    //can sort skus here to shown in product page if need
                    sort($skus);  //sort sku in ascending order;
                    foreach ($skus as $sku) {
                        $tempSourceList[$sku] = $this->getSourceData($sku);
                    }
                } else {
                    $tempSourceList[$sku] = $this->getSourceData($sku);
                }

                //Get MSI source_code for saleable associated simple products
                foreach ($tempSourceList as $sku => $sources) {

                    foreach ($sources as $source) {

                        if (!in_array($source->getData('source_code'), $sourceCodes)) {
                            $sourceCodes[] = $source->getData('source_code');
                        }
                    }
                }

                if (count($sourceCodes) > 1 && in_array("default", $sourceCodes)) { //set default store as first element
                    // Search
                    if (($key = array_search('default', $sourceCodes)) !== false) {
                        unset($sourceCodes[$key]);
                        array_unshift($sourceCodes, 'default');
                    }
                }

                //Form the MSI sources stock information for each saleable simple products. Such as default or birkenstock_newmarket
                foreach ($sourceCodes as $sourceCode) {
                    if (!isset($this->storeAllSourceList[$sourceCode])) {
                        continue;
                    }

                    $sourceData['source_code'] = $sourceCode;

                    if ($this->storeAllSourceList[$sourceCode]) {
                        $sourceData['source_info'] = $this->storeAllSourceList[$sourceCode];
                    }
                    $sourceData["sku_stock"] = array();
                    $sourceData["store_size"] = array();
                    $sourceData["stock_availability"] = array();  // 0 means sold out,  1 means low stock, 2 means In stock

                    $stockAvailability = array();

                    foreach ($tempSourceList as $sku => $sources) {
                        $skuExistSourceCode = array();
                        $skuSource = array();
                        $size = array();
                        $color = array();
                        $stockAvailability = array();
                        foreach ($sources as $source) {
                            //$this->msiArray[$i]["source"] = $sourceCode;
                            //get sku assigned MSI source
                            if (!in_array($source->getData('source_code'), $skuExistSourceCode)) {
                                $skuExistSourceCode[] = $source->getData('source_code');
                            }
                            //if sku assigned to the sourceCode, just fetch the stock value for the sku
                            if (in_array($sourceCode, $skuExistSourceCode)) {
                                if ($source->getData('source_code') == $sourceCode) { //get sku source qty
                                    $skuSource['sku'] = $sku;
                                    $skuSource['qty'] = $source->getData('quantity');
                                    if ($skuSource['qty'] <= 0) {
                                        $stockAvailability['stock_availability'] = "0";
                                    } else if ($skuSource['qty'] == 1) {
                                        $stockAvailability['stock_availability'] = "1";
                                    } else {
                                        $stockAvailability['stock_availability'] = "2";
                                    }
                                    foreach ($productList as $product) {
                                        if ($product->getSku() == $sku && $product->hasData("size")) {
                                            $skuSource['size'] = $product->getAttributeText("size");
                                            $size['size'] = $product->getAttributeText("size");
                                        }
                                    }
                                }
                            } else { // assign it to 0 if sku is not assigned to the sourceCode
                                $skuSource['sku'] = $sku;
                                $stockAvailability['stock_availability'] = 0;
                                $skuSource['qty'] = $source->getData('quantity');

                                foreach ($productList as $product) {
                                    if ($product->getSku() == $sku && $product->hasData("size")) {
                                        $skuSource['size'] = $product->getAttributeText("size");
                                        $size['size'] = $product->getAttributeText("size");
                                    }
                                }
                            }
                        }
                        if (!$sources) {
                            $skuSource['sku'] = $sku;
                            $stockAvailability['stock_availability'] = 0;
                            $skuSource['qty'] = 0;
                            foreach ($productList as $product) {
                                if ($product->getSku() == $sku && $product->hasData("size")) {
                                    $skuSource['size'] = $product->getAttributeText("size");
                                    $size['size'] = $product->getAttributeText("size");
                                }
                            }
                        }
                        $sourceData["sku_stock"][] = $skuSource;
                        $sourceData["store_size"][] = $size;
                        $sourceData["stock_availability"][] = $stockAvailability; // used to show availability in popup directly
                    }
                    $this->msiArray[$i]['store_size'] = $sourceData["store_size"];//// used to show size in popup directly
                    $this->msiArray[$i]['msi_lists'][] = $sourceData;
                }
            }
        }

    }

    /**
     * Get MSI basic inforamtion: product_id, attribute_code, option_title, sku
     * @param $attributeData
     * @return array
     */
    public function getMsiSkuDataIncludeOutStock($attributeData)
    {
        $msiArray = array();
        $widthArray = array();  // used to form width in the msi list data, it will be used when select color, it will contain all width for that color
        foreach ($attributeData as $attr) {
            foreach ($attr as $p) {
                $width = array();
                if (in_array($p['sku'], $this->allSimpleSku)) {
                    if (empty($msiArray)) {
                        $msi = array();
                        $msi['product_id'] = $p['product_id'];
                        $msi['attribute_code'] = $p['attribute_code'];
                        $msi['option_title'] = $p['option_title'];
                        $msi['sku'] = $p['sku'];
                        $msiArray[] = $msi;
                    } else {
                        $exist = false;
                        for ($i = 0; $i < count($msiArray); $i++) {
                            if ($p['product_id'] == $msiArray[$i]['product_id'] && $p['attribute_code'] == $msiArray[$i]['attribute_code'] && $p['attribute_code'] == "size") {  //if not width attribute but has size attribute, it will chain the size option directly. Size option has different option_tile
                                $msiArray[$i]['sku'] = $msiArray[$i]['sku'] . "," . $p['sku'];
                                $exist = true;
                                break;
                            }
                        }
                        if (!$exist) {
                            $msi = array();
                            $msi['product_id'] = $p['product_id'];
                            $msi['attribute_code'] = $p['attribute_code'];
                            $msi['option_title'] = $p['option_title'];
                            $msi['sku'] = $p['sku'];
                            $msiArray[] = $msi;
                        }
                    }
                }
            }
        }
        $this->msiArray = $msiArray;
        return $widthArray;
    }

    /**
     * @param $configProduct
     * @param $attributeData
     * @return array
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getAllSimpleProduct($configProduct, $attributeData)
    {
        $allSimpleSku = array();
        $options = $this->getAttributeOption($attributeData);
        
        if (empty($options)) {
            return $allSimpleSku;
        }
        
        // Extract all SKUs for batch loading
        $skus = array_keys($options);
        
        // Load all products in a single query using getList
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter('sku', $skus, 'in')
            ->create();
        $products = $this->productRepository->getList($searchCriteria);
        $productMap = [];
        
        // Create a map of SKU to product object for quick lookup
        foreach ($products->getItems() as $product) {
            $productMap[$product->getSku()] = $product;
        }
        
        // Process each SKU using the pre-loaded data
        foreach ($options as $sku => $optionTitle) {
            if (isset($productMap[$sku])) {
                $product = $productMap[$sku];
                $allSimpleSku[] = $product;
                $this->allSimpleSku[] = $sku;
            }
        }
        
        return $allSimpleSku;
    }

    /**
     * Generate msi Stock Data after generate msi data
     * get first msi stock as init msi stock
     */
    public function generateInitMsiStock($cid)
    {
        foreach ($this->msiArray as $msi) {
            if ($msi['product_id'] == $cid) {
                $this->msiStock = $msi['msi_lists'];
                $this->msiStoreSize = $msi['store_size'];
                break;  //get first msi stock as init msi stock
            }
        }
    }


    /**
     * Get All source list
     *
     * @return SourceInterface[]|null
     */
    public function getStoreAllSourcesList()
    {
        try {
            $items = $this->sourcesService->execute();
            if ($items) {
                foreach ($items as $item) {
                    $storeInfo = [];
                    if ($item->hasData('source_code')) {
                        $storeInfo['source_name'] = $item->getData('name');
                        $storeInfo['source_address'] = $item->getData('street');
                        $storeInfo['source_city'] = $item->getData('city');
                        $storeInfo['source_phone'] = $item->getData('phone');
                        $this->storeAllSourceList[$item->getData('source_code')] = $storeInfo;
                    }
                }
            }
        } catch (Exception $exception) {
            $this->logger->error($exception->getMessage());
        }
        return null;
    }


    /**
     *  Get MSI Source Item  msiData
     */
    public function getSourceItemBySku($sku)
    {
        return $this->sourceItemService->execute($sku);
    }

    /**
     *  Get MSI Source Data by Sku
     */
    public function getSourceData($sku)
    {
        $sources = array();
        $sourceItemList = $this->getSourceItemBySku($sku);

        return $sourceItemList;
    }
}
