<?php

namespace Overdose\InStorePickup\Model\Service;

use Magento\Framework\Exception\LocalizedException;
use Magento\Inventory\Model\ResourceModel\SourceItem\CollectionFactory;
use Magento\Inventory\Model\ResourceModel\SourceItem\Collection;
use Magento\Inventory\Model\ResourceModel\Source;
use Magento\Inventory\Model\ResourceModel\StockSourceLink;
use Magento\InventorySalesApi\Model\GetAssignedStockIdForWebsiteInterface;
use Magento\Store\Model\StoreManagerInterface;

class SourceItemService
{
    /**
     * @var CollectionFactory
     */
    private $collectionFactory;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var GetAssignedStockIdForWebsiteInterface
     */
    private $assignedStock;

    /**
     * SourcesService constructor.
     * @param CollectionFactory $collectionFactory
     */
    public function __construct(
        StoreManagerInterface $storeManager,
        GetAssignedStockIdForWebsiteInterface $assignedStock,
        CollectionFactory $collectionFactory
    ) {
        $this->collectionFactory = $collectionFactory;
        $this->storeManager = $storeManager;
        $this->assignedStock = $assignedStock;
    }

    /**
     * Returns sources by sku
     * @param string|null $websiteCode
     * @return \Magento\Framework\DataObject[]
     * @throws LocalizedException
     */
    public function execute(string $sku): array
    {
        $collection = $this->collectionFactory->create();
        $collection->addFieldToFilter('sku', $sku);
        $this->joinInventorySource($collection);
        $this->joinInventoryLink($collection);
        return $collection->getItems();
    }

    /**
     * @param Collection $collection
     * @return void
     */
    private function joinInventorySource(Collection $collection)
    {
        $collection->getSelect()->joinLeft(
            [
                'source_table' => $collection->getTable(
                    Source::TABLE_NAME_SOURCE
                ),
            ],
            '`main_table`.`source_code` = `source_table`.`source_code`',
            ['enabled' => 'source_table.enabled']
        )->where("`source_table`.`enabled` = '1'");
    }

    /**
     * @param Collection $collection
     * @return void
     */
    private function joinInventoryLink(Collection $collection)
    {

        $websiteCode = $this->storeManager->getWebsite()->getCode();
        $stockId = (int)$this->assignedStock->execute($websiteCode);

        $collection->getSelect()->joinLeft(
            [
                'link_table' => $collection->getTable(
                    StockSourceLink::TABLE_NAME_STOCK_SOURCE_LINK
                ),
            ],
            '`main_table`.`source_code` = `link_table`.`source_code`',
            [
                'priority' => 'link_table.priority',
                'stock_id' => 'link_table.stock_id'
            ]
        )->where('link_table.stock_id = ?', $stockId)
            ->order('link_table.priority ASC');

    }
}
