<?php
/**
 * Copyright © Overdose Digital. All rights reserved.
 */
namespace Overdose\EcoShipping\Model\Carrier;

use Magento\Quote\Model\Quote\Address\RateRequest;
use Magento\Shipping\Model\Carrier\AbstractCarrier;
use Magento\Shipping\Model\Carrier\CarrierInterface;
use Magento\Shipping\Model\Rate\Result;

/**
 * Eco-friendly signature required shipping method implementation
 */
class EcoSignature extends AbstractCarrier implements CarrierInterface
{
    /**
     * @var string
     */
    protected $_code = 'ecosignature';

    /**
     * @var bool
     */
    protected $_isFixed = true;

    /**
     * @var \Magento\Shipping\Model\Rate\ResultFactory
     */
    protected $_rateResultFactory;

    /**
     * @var \Magento\Quote\Model\Quote\Address\RateResult\MethodFactory
     */
    protected $_rateMethodFactory;

    /**
     * @var \Magento\Customer\Model\Session
     */
    protected $customerSession;

    /**
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     * @param \Magento\Quote\Model\Quote\Address\RateResult\ErrorFactory $rateErrorFactory
     * @param \Psr\Log\LoggerInterface $logger
     * @param \Magento\Shipping\Model\Rate\ResultFactory $rateResultFactory
     * @param \Magento\Quote\Model\Quote\Address\RateResult\MethodFactory $rateMethodFactory
     * @param \Magento\Customer\Model\Session $customerSession
     * @param array $data
     */
    public function __construct(
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Quote\Model\Quote\Address\RateResult\ErrorFactory $rateErrorFactory,
        \Psr\Log\LoggerInterface $logger,
        \Magento\Shipping\Model\Rate\ResultFactory $rateResultFactory,
        \Magento\Quote\Model\Quote\Address\RateResult\MethodFactory $rateMethodFactory,
        \Magento\Customer\Model\Session $customerSession,
        array $data = []
    ) {
        $this->_rateResultFactory = $rateResultFactory;
        $this->_rateMethodFactory = $rateMethodFactory;
        $this->customerSession = $customerSession;
        parent::__construct($scopeConfig, $rateErrorFactory, $logger, $data);
    }

    /**
     * Collect and get rates
     *
     * @param RateRequest $request
     * @return Result|bool
     */
    public function collectRates(RateRequest $request)
    {
        if (!$this->getConfigFlag('active')) {
            return false;
        }

        // Check if postcode starts with 7, 8, or 9
        $postcode = $request->getDestPostcode();
        if (!$this->isValidPostcode($postcode)) {
            return false;
        }

        /** @var Result $result */
        $result = $this->_rateResultFactory->create();

        // Add eco-friendly signature required method
        $method = $this->_rateMethodFactory->create();

        $method->setCarrier($this->_code);
        $method->setCarrierTitle($this->getConfigData('title'));

        $method->setMethod($this->_code);
        $method->setMethodTitle($this->getConfigData('name'));

        $price = $this->getConfigData('price');
        
        $forceFreeShippingAmount = $this->getForceFreeShippingAmount();        
        if (($forceFreeShippingAmount != null && $request->getPackageValueWithDiscount() > (float)$forceFreeShippingAmount) 
            || $this->customerSession->isLoggedIn()) {
            $price = 0;
        }
        
        $method->setPrice($price);
        $method->setCost($price);

        $result->append($method);

        return $result;
    }

    /**
     * Check if postcode starts with 7, 8, or 9
     *
     * @param string $postcode
     * @return bool
     */
    private function isValidPostcode($postcode)
    {
        if (empty($postcode)) {
            return false;
        }

        $firstDigit = substr(trim($postcode), 0, 1);
        return in_array($firstDigit, ['7', '8', '9']);
    }

    /**
     * Get allowed shipping methods
     *
     * @return array
     */
    public function getAllowedMethods()
    {
        return [
            'ecosignature' => $this->getConfigData('name')
        ];
    }
    
    /**
     * @return mixed
     */
    public function getForceFreeShippingAmount()
    {
        return $this->getConfigData('force_free_shipping');
    }
}
