<?php
/**
 * BrittainWynyard GiftcardAccount Body Class Observer
 *
 * @package BrittainWynyard_GiftcardAccount
 */

namespace BrittainWynyard\GiftcardAccount\Observer;

use BrittainWynyard\GiftcardAccount\Helper\Config;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\App\Request\Http as HttpRequest;
use Magento\Framework\View\Page\Config as PageConfig;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Checkout\Model\Session as CheckoutSession;

/**
 * Observer to add gift card visibility CSS classes to body element
 */
class AddGiftcardBodyClassObserver implements ObserverInterface
{
    /**
     * @var Config
     */
    private $config;

    /**
     * @var PageConfig
     */
    private $pageConfig;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var HttpRequest
     */
    private $request;

    /**
     * @var CheckoutSession
     */
    private $checkoutSession;

    /**
     * @param Config $config
     * @param PageConfig $pageConfig
     * @param StoreManagerInterface $storeManager
     * @param HttpRequest $request
     * @param CheckoutSession $checkoutSession
     */
    public function __construct(
        Config $config,
        PageConfig $pageConfig,
        StoreManagerInterface $storeManager,
        HttpRequest $request,
        CheckoutSession $checkoutSession
    ) {
        $this->config = $config;
        $this->pageConfig = $pageConfig;
        $this->storeManager = $storeManager;
        $this->request = $request;
        $this->checkoutSession = $checkoutSession;
    }

    /**
     * Add gift card visibility CSS class to body
     *
     * @param Observer $observer
     * @return void
     */
    public function execute(Observer $observer): void
    {      
        // Only add classes on checkout and cart pages
        if ($this->isCheckoutOrCartPage()) {
            $storeId = $this->storeManager->getStore()->getId();
            
            // Hide gift card payment option if:
            // 1. Configuration says it should be hidden, OR
            // 2. Cart contains gift card products
            $shouldHide = false;
            
            if (!$this->config->shouldGiftcardBeVisible($storeId)) {
                $shouldHide = true;
            } elseif ($this->hasGiftCardProductInCart()) {
                $shouldHide = true;
            }
            
            if ($shouldHide) {
                $this->pageConfig->addBodyClass('bw-giftcard-hidden');
            }
        }
    }

    /**
     * Check if current page is checkout or cart
     *
     * @return bool
     */
    private function isCheckoutOrCartPage(): bool
    {
        $routeName = $this->request->getRouteName();
        $controllerName = $this->request->getControllerName();
        $actionName = $this->request->getActionName();
        
        $fullActionName = $routeName . '_' . $controllerName . '_' . $actionName;
        
        return in_array($fullActionName, [
            'checkout_index_index',
            'checkout_cart_index'
        ]);
    }

    /**
     * Check if cart contains any gift card products
     *
     * @return bool
     */
    private function hasGiftCardProductInCart(): bool
    {
        try {
            $quote = $this->checkoutSession->getQuote();
            
            if (!$quote || !$quote->getId()) {
                return false;
            }
            
            $items = $quote->getAllVisibleItems();
            
            foreach ($items as $item) {
                if ($item->getProductType() === 'giftcard') {
                    return true;
                }
            }
            
            return false;
        } catch (\Exception $e) {
            // If there's any error checking the cart, default to not hiding
            return false;
        }
    }
}
