<?php

declare(strict_types=1);

namespace BrittainWynyard\CatalogSuperStyle\Helper;

use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Helper\Context;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Framework\App\ResourceConnection;
use Psr\Log\LoggerInterface;
use BrittainWynyard\CatalogSuperStyle\Model\ResourceModel\Index\CollectionFactory;

class Data extends AbstractHelper
{
    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var ResourceConnection
     */
    private $resourceConnection;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var CollectionFactory
     */
    private $collectionFactory;

    /**
     * @var array
     */
    private $superSwatchCounts = [];

    /**
     * Constructor
     *
     * @param Context $context
     * @param StoreManagerInterface $storeManager
     * @param ResourceConnection $resourceConnection
     * @param LoggerInterface $logger
     * @param CollectionFactory $collectionFactory
     */
    public function __construct(
        Context $context,
        StoreManagerInterface $storeManager,
        ResourceConnection $resourceConnection,
        LoggerInterface $logger,
        CollectionFactory $collectionFactory
    ) {
        parent::__construct($context);
        $this->storeManager = $storeManager;
        $this->resourceConnection = $resourceConnection;
        $this->logger = $logger;
        $this->collectionFactory = $collectionFactory;
    }

    /**
     * Load super swatch counts for multiple products
     *
     * @param array $products
     * @return void
     */
    public function loadSwatchCounts(array $products): void
    {
        // Extract unique super_style_ids from products and set default count for all products
        $superStyleIds = [];
        $productsWithoutSuperStyleId = [];
        
        foreach ($products as $product) {
            $superStyleId = $product->getData('super_style_id');
            if ($superStyleId) {
                $superStyleIds[] = $superStyleId;
            } else {
                // Products without super_style_id should have count of 1
                $product->setData('super_swatch_related_count', 1);
                $productsWithoutSuperStyleId[] = $product;
            }
        }

        if (empty($superStyleIds)) {
            return; // All products have been processed (those without super_style_id)
        }

        $superStyleIds = array_unique($superStyleIds);
        
        // Filter out style IDs we've already loaded
        $styleIdsToLoad = [];
        foreach ($superStyleIds as $styleId) {
            if (!isset($this->superSwatchCounts[$styleId])) {
                $styleIdsToLoad[] = $styleId;
            }
        }
        
        if (empty($styleIdsToLoad)) {
            return; // All requested style IDs are already loaded
        }

        // Load counts from the index table
        $this->loadMissingStyleCounts($styleIdsToLoad);

        // Apply counts to products
        foreach ($products as $product) {
            $superStyleId = $product->getData('super_style_id');
            if ($superStyleId && isset($this->superSwatchCounts[$superStyleId])) {
                $product->setData('super_swatch_related_count', $this->superSwatchCounts[$superStyleId]);
            }
        }
    }

    /**
     * Load counts from index table (super fast!)
     *
     * @param array $styleIdsToLoad
     */
    private function loadMissingStyleCounts(array $styleIdsToLoad): void
    {
        if (empty($styleIdsToLoad)) {
            return;
        }

        try {
            $websiteId = (int)$this->storeManager->getWebsite()->getId();
            
            // Use ORM Collection for COUNT query on indexed data
            $collection = $this->collectionFactory->create();
            $results = $collection->getCountsByStyleIds($styleIdsToLoad, $websiteId);

            // Populate counts from index
            foreach ($styleIdsToLoad as $styleId) {
                $this->superSwatchCounts[$styleId] = (int)($results[$styleId] ?? 1);
            }

        } catch (\Exception $e) {
            $this->logger->error('Data: Failed to load super swatch counts from index', [
                'error' => $e->getMessage(),
                'style_ids_count' => count($styleIdsToLoad)
            ]);
            
            // Set default counts to 1 for all failed lookups
            foreach ($styleIdsToLoad as $styleId) {
                $this->superSwatchCounts[$styleId] = 1;
            }
        }
    }

    /**
     * Get product IDs for super style IDs (new feature!)
     *
     * @param array $superStyleIds
     * @return array
     */
    public function getProductIdsBySuperstyleIds(array $superStyleIds): array
    {
        if (empty($superStyleIds)) {
            return [];
        }

        try {
            $websiteId = (int)$this->storeManager->getWebsite()->getId();

            // Use ORM Collection for GROUP_CONCAT query on indexed data
            $collection = $this->collectionFactory->create();
            $results = $collection->getProductIdsByStyleIds($superStyleIds, $websiteId);

            // Convert to arrays
            $productIdsByStyleId = [];
            foreach ($results as $styleId => $productIds) {
                $productIdsByStyleId[$styleId] = explode(',', $productIds);
            }

            return $productIdsByStyleId;

        } catch (\Exception $e) {
            $this->logger->error('Data: Failed to load product IDs from index', [
                'error' => $e->getMessage(),
                'super_style_ids' => $superStyleIds
            ]);
            return [];
        }
    }


    /**
     * Get the cached super swatch count for a product
     *
     * @param string $superStyleId
     * @return int
     */
    public function getSuperSwatchCount(string $superStyleId): int
    {
        return (int)($this->superSwatchCounts[$superStyleId] ?? 0);
    }

    /**
     * Get Super Swatch Count for a single product (improved)
     *
     * @param mixed $product
     * @return int
     */
    public function getSuperSwatchCountForProduct($product): int
    {
        $superStyleId = $product->getData('super_style_id');
        
        if (!$superStyleId) {
            return 1;
        }
        
        if (!isset($this->superSwatchCounts[$superStyleId])) {
            $this->loadMissingStyleCounts([$superStyleId]);
        }
        
        return (int)($this->superSwatchCounts[$superStyleId] ?? 1);
    }

    /**
     * Reset the cache
     */
    public function reset(): void
    {
        $this->superSwatchCounts = [];
    }
}
