/*
 * Copyright © Overdose Digital. All rights reserved.
 * See LICENSE_OVERDOSE.txt for license details.
 */
define([
    'Magento_Ui/js/modal/alert',
    'jquery',
    'Magento_Checkout/js/model/quote',
    'Magento_Customer/js/customer-data',
    'mage/url',
    'Magento_Checkout/js/model/cart/totals-processor/default',
    'Magento_Ui/js/modal/confirm',
    'jquery-ui-modules/widget',
    'mage/validation',
    'mage/translate',
    'mage/cookies'
], function (alert, $, quote, customerData, urlBuilder, totalsDefaultProvider, confirmation) {
    'use strict';

    var updateShoppingCartMixin = {
        options: {
            updateURL: 'checkout/sidebar/updateItemQty',
            removeURL: 'checkout/sidebar/removeItem',
            updateEventName: 'updateCartItemQty',
            removeEventName: 'removeFromCart',
            cartFormEl: '.cart-wrapper.form-cart',
            reloadSections: ['checkout.cart.form', 'checkout.cart.methods.bottom'],
            sectionElementAssoc: [
                {sectionName: 'checkout.cart.form', selector: '.form-cart', parent: '.cart-container'},
                {sectionName: 'checkout.cart.methods.bottom', selector: '.checkout-methods-items', parent: '.cart-summary'},
            ],
            updateDelay: 1500
        },

        /** @inheritdoc */
        _create: function () {
            this._super();

            this.initBindItems();
        },

        /**
         * Initialize bindings for items
         */
        initBindItems: function () {
            var self = this,
                timerId = null,
                qtyEl = $(this.options.cartFormEl).find('.qty'),
                deleteEl = $('.action-delete');

            qtyEl.on('change', function (e) {
                clearTimeout(timerId);
                timerId = setTimeout(function () {
                    self.updateQty(e);
                }, self.options.updateDelay);
            });

            deleteEl.unbind();
            deleteEl.on('click', function (el) {
                el.stopImmediatePropagation();

                var itemId = $(this).data('post').data.id;
                confirmation({
                    title: $.mage.__('Remove item'),
                    content: $.mage.__('Remove product from the cart?'),
                    actions: {
                        confirm: function () {
                            self.sendRequest(
                                urlBuilder.build(self.options.removeURL),
                                {
                                    'item_id': itemId,
                                    'reload_sections': self.options.reloadSections
                                },
                                self.options.removeEventName
                            );
                        }
                    }
                });

                return false;
            });
        },

        /**
         * Prevents default submit action and calls form validator.
         *
         * @param {Event} event
         *
         * @return {Boolean}
         */
        updateQty: function (event) {
            if (!this.options.validationURL && !this.options.updateQtyURL) {
                return true;
            }
            event.preventDefault();

            if (this.isValid()) {
                var itemId = this._getItemId($(event.target).attr('name')),
                    qty    = $(event.target).val();

                this.sendRequest(
                    urlBuilder.build(this.options.updateURL),
                    {
                        'item_id': itemId,
                        'item_qty': qty,
                        'reload_sections': this.options.reloadSections
                    },
                    this.options.updateEventName
                );
            }
        },

        /**
         * Get item ID
         *
         * @param qtyName
         *
         * @returns {null|*}
         * @private
         */
        _getItemId: function (qtyName) {
            var parsedValue = qtyName.match(/(.*)\[(\d+)\]\[(.*)]/);
            if (parsedValue.length === 4) {
                return parsedValue[2];
            }
            return null;
        },

        /**
         * Update shopping cart data.
         *
         * @param {String} url - request url
         * @param {Object} data - post data for ajax call
         * @param {String} eventName - event name
         */
        sendRequest: function (url, data, eventName) {
            $.extend(data, {
                'form_key': $.mage.cookies.get('form_key')
            });

            $.ajax({
                url: url,
                data: data,
                type: 'POST',
                dataType: 'json',
                context: this,

                /** @inheritdoc */
                beforeSend: function () {
                    $(document.body).trigger('processStart');
                },

                /** @inheritdoc */
                complete: function () {
                    $(document.body).trigger('processStop');
                }
            }).done(function (response) {
                if (response.success) {
                    let eventData = {},
                        searchCriteria,
                        product;
                    if (eventName === this.options.removeEventName) {
                        searchCriteria = function (item) {
                            if (item['item_id'] !== data.item_id) {
                                return false;
                            }
                            return true;
                        };
                        product = _.find(customerData.get('cart')().items, searchCriteria)

                        eventData = {
                            productIds: [
                                product.product_id
                            ],
                            productInfo: [
                                {
                                    'id': product.product_id,
                                    'optionValues': product.options[0].option_value
                                }
                            ]
                        };
                    }

                    this.onSuccess(response, eventName, eventData);
                } else {
                    this.onError(response);
                    $(this.options.cartFormEl).trigger('reset');
                }
            }).fail(function () {
                this.submitForm();
            });
        },

        /**
         * Prevents default submit action and calls form validator.
         *
         * @param {Event} event
         * @return {Boolean}
         */
        onSubmit: function (event) {
            event.preventDefault();
        },

        /**
         * Form validation succeed.
         *
         * @param response
         * @param eventName
         * @param eventData - the data which will be sent with event trigger, must be in object format: {}
         */
        onSuccess: function (response, eventName, eventData) {
            $(document).trigger('ajax:' + eventName, eventData);

            totalsDefaultProvider.estimateTotals(quote.shippingAddress());
            if (response.reload_page) {
                window.location.reload();
            } else {
                if (response.sections) {
                    this.updateSections(response.sections);
                }
            }
        },

        /**
         * Update page. Replace HTML with new content.
         * Create block if not exist.
         *
         * @param {array} sections
         */
        updateSections: function (sections) {
            for (var name in sections) {
                let section = this.options.sectionElementAssoc.find(obj => {
                    return obj.sectionName === name
                });
                if (section) {
                    let content = sections[name];
                    let targetElement = $(section.selector);
                    if (targetElement.length) {
                        // replace existing block
                        targetElement.replaceWith(content).trigger('contentUpdated');
                    } else {
                        console.warn('Element doesn\'t exist: %s', section.selector);
                        // apeend to parent if not exist before
                        let parent = $(section.parent);
                        if (parent.length) {
                            parent.append(content).trigger('contentUpdated');
                        } else {
                            console.warn('Element doesn\'t exist: %s', section.parent);
                        }
                    }
                } else {
                    console.warn('Undefined section: %s', name);
                }
            }

            if ($.fn.applyBindings !== undefined) {
               $(this.options.cartFormEl).applyBindings();
            }
        }
    };

    return function (targetWidget) {
        $.widget('mage.updateShoppingCart', targetWidget, updateShoppingCartMixin);

        return $.mage.updateShoppingCart;
    };
});

