<?php

namespace Overdose\ShoeFinder\Model\Config\Source;

use Magento\Framework\Module\ModuleListInterface;
use Magento\Framework\Module\Dir\Reader as ModuleReader;
use Magento\Framework\App\Filesystem\DirectoryList;

/**
 * Source model for controller actions
 */
class ControllerActions implements \Magento\Framework\Option\ArrayInterface
{
    /**
     * @var ModuleListInterface
     */
    private $moduleList;

    /**
     * @var ModuleReader
     */
    private $moduleReader;

    /**
     * @var DirectoryList
     */
    private $directoryList;

    /**
     * @var array
     */
    private $options = null;

    /**
     * @var array
     */
    private $routeMap = null;

    /**
     * @param ModuleListInterface $moduleList
     * @param ModuleReader $moduleReader
     * @param DirectoryList $directoryList
     */
    public function __construct(
        ModuleListInterface $moduleList,
        ModuleReader $moduleReader,
        DirectoryList $directoryList
    ) {
        $this->moduleList = $moduleList;
        $this->moduleReader = $moduleReader;
        $this->directoryList = $directoryList;
    }

    /**
     * Return array of options as value-label pairs
     *
     * @return array
     */
    public function toOptionArray()
    {
        if ($this->options === null) {
            $this->options = [];
            $actions = $this->getAllControllerActions();
            
            // Actions are already filtered to frontend only
            
            // Sort alphabetically
            sort($actions);
            
            foreach ($actions as $action) {
                $this->options[] = [
                    'value' => $action,
                    'label' => $action
                ];
            }
        }

        return $this->options;
    }

    /**
     * Get all controller actions from enabled modules
     *
     * @return array
     */
    private function getAllControllerActions()
    {
        $actions = [];
        $enabledModules = $this->moduleList->getNames();

        foreach ($enabledModules as $moduleName) {
            try {
                $moduleActions = $this->getModuleControllerActions($moduleName);
                $actions = array_merge($actions, $moduleActions);
            } catch (\Exception $e) {
                // Skip modules that don't have controllers or have issues
                continue;
            }
        }

        return array_unique($actions);
    }

    /**
     * Get controller actions for a specific module
     *
     * @param string $moduleName
     * @return array
     */
    private function getModuleControllerActions($moduleName)
    {
        $actions = [];
        
        try {
            $moduleDir = $this->moduleReader->getModuleDir('', $moduleName);
            $controllerDir = $moduleDir . '/Controller';
            
            if (!is_dir($controllerDir)) {
                return $actions;
            }

            // Get route name from frontend routes.xml or fallback to module name conversion
            $routeName = $this->getRouteNameFromModule($moduleName, 'frontend');
            
            // Only scan frontend controller directories
            $frontendControllerDir = $controllerDir . '/Frontend';
            
            if (is_dir($frontendControllerDir)) {
                $this->scanControllerDirectory($frontendControllerDir, $routeName, '', $actions);
            } else {
                // For modules that don't use Frontend/Adminhtml structure, 
                // only scan if it's a frontend route (has frontend routes.xml)
                if ($this->hasFrontendRoute($moduleName)) {
                    $this->scanControllerDirectory($controllerDir, $routeName, '', $actions);
                }
            }
        } catch (\Exception $e) {
            // Module might not exist or have issues
        }

        return $actions;
    }

    /**
     * Recursively scan controller directory
     *
     * @param string $dir
     * @param string $routeName
     * @param string $subPath
     * @param array &$actions
     * @return void
     */
    private function scanControllerDirectory($dir, $routeName, $subPath, &$actions)
    {
        if (!is_dir($dir)) {
            return;
        }

        $items = scandir($dir);
        
        foreach ($items as $item) {
            if ($item === '.' || $item === '..') {
                continue;
            }

            $itemPath = $dir . '/' . $item;
            
            if (is_dir($itemPath)) {
                // Skip adminhtml directory
                if (strtolower($item) === 'adminhtml') {
                    continue;
                }
                // Recursively scan subdirectories
                $newSubPath = $subPath ? $subPath . '_' . strtolower($item) : strtolower($item);
                $this->scanControllerDirectory($itemPath, $routeName, $newSubPath, $actions);
            } elseif (is_file($itemPath) && pathinfo($item, PATHINFO_EXTENSION) === 'php') {
                // Found a controller file
                $fileName = pathinfo($item, PATHINFO_FILENAME);
                $actionInfo = $this->extractActionInfo($itemPath, $fileName);
                
                if ($actionInfo) {
                    $fullActionName = $routeName;
                    
                    // Build controller path from subPath
                    if ($subPath) {
                        $fullActionName .= '_' . $subPath;
                    }
                    
                    // For modern Magento 2 controllers with execute(), the class name is the action
                    // and the directory path is the controller
                    if ($actionInfo['isExecuteMethod']) {
                        // Controller is the subPath, action is the class name
                        $fullActionName .= '_' . $actionInfo['actionName'];
                    } else {
                        // Legacy controllers: controller is filename, action is method name
                        $fullActionName .= '_' . strtolower($fileName) . '_' . $actionInfo['actionName'];
                    }
                    
                    $actions[] = $fullActionName;
                }
            }
        }
    }

    /**
     * Extract action information from controller file
     *
     * @param string $filePath
     * @param string $fileName
     * @return array|null Returns array with 'actionName' and 'isExecuteMethod', or null
     */
    private function extractActionInfo($filePath, $fileName)
    {
        $content = @file_get_contents($filePath);
        if (!$content) {
            return null;
        }

        // For modern Magento 2 controllers with execute() method
        // The action name is the class name (e.g., View class -> view action)
        if (preg_match('/public\s+function\s+execute\s*\(/', $content)) {
            if (preg_match('/class\s+(\w+)\s+extends/', $content, $matches)) {
                $className = $matches[1];
                // Convert PascalCase to lowercase (e.g., Index -> index, ProductView -> product_view)
                $actionName = strtolower(preg_replace('/([a-z])([A-Z])/', '$1_$2', $className));
                return [
                    'actionName' => $actionName,
                    'isExecuteMethod' => true
                ];
            }
        }

        // Look for legacy action methods (e.g., public function indexAction())
        if (preg_match('/public\s+function\s+(\w+)Action\s*\(/', $content, $matches)) {
            return [
                'actionName' => strtolower($matches[1]),
                'isExecuteMethod' => false
            ];
        }

        return null;
    }

    /**
     * Convert module name to route name
     * First tries to read from frontend routes.xml, then falls back to conversion
     *
     * @param string $moduleName
     * @param string $area
     * @return string
     */
    private function getRouteNameFromModule($moduleName, $area = 'frontend')
    {
        // Build route map if not already built
        if ($this->routeMap === null) {
            $this->buildRouteMap();
        }
        
        // Check if we have a route mapping for this module
        if (isset($this->routeMap[$moduleName])) {
            return $this->routeMap[$moduleName];
        }
        
        // Fallback to conversion
        $parts = explode('_', $moduleName);
        if (count($parts) >= 2) {
            // Take the second part and convert to lowercase
            $routeName = strtolower($parts[1]);
            // Handle camelCase conversion (e.g., ShoeFinder -> shoefinder)
            $routeName = preg_replace('/([a-z])([A-Z])/', '$1$2', $routeName);
            $routeName = strtolower($routeName);
            return $routeName;
        }
        
        return strtolower(str_replace('_', '', $moduleName));
    }

    /**
     * Build route map from frontend routes.xml files only
     *
     * @return void
     */
    private function buildRouteMap()
    {
        $this->routeMap = [];
        $enabledModules = $this->moduleList->getNames();
        
        foreach ($enabledModules as $moduleName) {
            try {
                // Only check frontend routes
                $frontendRoutes = $this->readRoutesXml($moduleName, 'frontend');
                
                if (!empty($frontendRoutes)) {
                    // Use the first frontName found
                    $this->routeMap[$moduleName] = $frontendRoutes[0];
                }
            } catch (\Exception $e) {
                // Skip modules without frontend routes.xml
                continue;
            }
        }
    }

    /**
     * Check if module has a frontend route
     *
     * @param string $moduleName
     * @return bool
     */
    private function hasFrontendRoute($moduleName)
    {
        if ($this->routeMap === null) {
            $this->buildRouteMap();
        }
        
        return isset($this->routeMap[$moduleName]);
    }

    /**
     * Read routes.xml file for a module
     *
     * @param string $moduleName
     * @param string $area
     * @return array
     */
    private function readRoutesXml($moduleName, $area = 'frontend')
    {
        $routes = [];
        
        try {
            $moduleDir = $this->moduleReader->getModuleDir('etc', $moduleName);
            $routesFile = $moduleDir . '/' . $area . '/routes.xml';
            
            if (!file_exists($routesFile)) {
                return $routes;
            }
            
            $xml = @simplexml_load_file($routesFile);
            if ($xml === false) {
                return $routes;
            }
            
            // Parse routes
            foreach ($xml->router as $router) {
                foreach ($router->route as $route) {
                    $frontName = (string)$route['frontName'];
                    if ($frontName) {
                        $routes[] = $frontName;
                    }
                }
            }
        } catch (\Exception $e) {
            // File might not exist or be unreadable
        }
        
        return $routes;
    }
}

