<?php

namespace Overdose\Seo\Observer;

use Magento\Bundle\Model\Product\Type as BundleType;
use Magento\Bundle\Model\Product\TypeFactory;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Model\Product;
use Magento\Catalog\Model\Product\Attribute\Source\Status;
use Magento\Catalog\Model\Product\Type;
use Magento\Catalog\Model\ResourceModel\Category\CollectionFactory as CatalogCollectionFactory;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory;
use Magento\ConfigurableProduct\Model\ResourceModel\Product\Type\Configurable;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Registry;
use Magento\Framework\UrlInterface;
use Magento\Framework\View\Element\Template\Context;
use Magento\GroupedProduct\Model\Product\Type\Grouped;
use Magento\Store\Model\StoreManagerInterface;
use Magento\UrlRewrite\Model\ResourceModel\UrlRewriteCollection;
use Mirasvit\Seo\Api\Service\CanonicalRewrite\CanonicalRewriteServiceInterface;
use Mirasvit\Seo\Api\Service\StateServiceInterface;
use Mirasvit\Seo\Helper\Data;
use Mirasvit\Seo\Helper\UrlPrepare;
use Mirasvit\Seo\Model\Config;

/**
 * @SuppressWarnings(PHPMD)
 */
class Canonical implements ObserverInterface
{
    /**
     * @var Config
     */
    protected $config;

    /**
     * @var ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * @var TypeFactory
     */
    protected $productTypeFactory;

    /**
     * @var CatalogCollectionFactory
     */
    protected $categoryCollectionFactory;

    /**
     * @var CollectionFactory
     */
    protected $productCollectionFactory;

    /**
     * @var Context
     */
    protected $context;

    /**
     * @var Registry
     */
    protected $registry;

    /**
     * @var Data
     */
    protected $seoData;

    /**
     * @var StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var RequestInterface
     */
    protected $request;

    /**
     * @var Configurable
     */
    protected $productTypeConfigurable;

    /**
     * @var BundleType
     */
    protected $productTypeBundle;

    /**
     * @var Grouped
     */
    protected $productTypeGrouped;

    /**
     * @var UrlRewriteCollection
     */
    protected $urlRewrite;

    /**
     * @var UrlPrepare
     */
    protected $urlPrepare;

    /**
     * @var string
     */
    private $fullAction;

    /**
     * @var ProductRepositoryInterface
     */
    private $productRepository;

    /**
     * @var StateServiceInterface
     */
    private $stateService;

    /**
     * @var CanonicalRewriteServiceInterface
     */
    private $canonicalRewriteService;

    /**
     * @param Config $config
     * @param TypeFactory $productTypeFactory
     * @param CatalogCollectionFactory $categoryCollectionFactory
     * @param CollectionFactory $productCollectionFactory
     * @param Configurable $productTypeConfigurable
     * @param BundleType $productTypeBundle
     * @param Grouped $productTypeGrouped
     * @param Context $context
     * @param Registry $registry
     * @param Data $seoData
     * @param UrlRewriteCollection $urlRewrite
     * @param UrlPrepare $urlPrepare
     * @param CanonicalRewriteServiceInterface $canonicalRewriteService
     * @param ProductRepositoryInterface $productRepository
     * @param StateServiceInterface $stateService
     */
    public function __construct(
        Config $config,
        TypeFactory $productTypeFactory,
        CatalogCollectionFactory $categoryCollectionFactory,
        CollectionFactory $productCollectionFactory,
        Configurable $productTypeConfigurable,
        BundleType $productTypeBundle,
        Grouped $productTypeGrouped,
        Context $context,
        Registry $registry,
        Data $seoData,
        UrlRewriteCollection $urlRewrite,
        UrlPrepare $urlPrepare,
        CanonicalRewriteServiceInterface $canonicalRewriteService,
        ProductRepositoryInterface $productRepository,
        StateServiceInterface $stateService
    ) {
        $this->config                    = $config;
        $this->productTypeFactory        = $productTypeFactory;
        $this->categoryCollectionFactory = $categoryCollectionFactory;
        $this->productCollectionFactory  = $productCollectionFactory;
        $this->productTypeConfigurable   = $productTypeConfigurable;
        $this->productTypeBundle         = $productTypeBundle;
        $this->productTypeGrouped        = $productTypeGrouped;
        $this->context                   = $context;
        $this->registry                  = $registry;
        $this->seoData                   = $seoData;
        $this->storeManager              = $context->getStoreManager();
        $this->request                   = $context->getRequest();
        $this->fullAction                = $this->request->getFullActionName();
        $this->urlRewrite                = $urlRewrite;
        $this->urlPrepare                = $urlPrepare;
        $this->canonicalRewriteService   = $canonicalRewriteService;
        $this->productRepository         = $productRepository;
        $this->stateService              = $stateService;
    }

    /**
     * @param Observer $observer
     *
     * @return void
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function execute(Observer $observer)
    {
        $this->fullAction = $observer->getRequest()->getFullActionName();
        if ($this->fullAction !== '__') {
            $this->setupCanonicalUrl();
        }
    }

    /**
     * @return bool
     */
    public function setupCanonicalUrl()
    {
        if ($this->seoData->isIgnoredActions()
            && !$this->seoData->cancelIgnoredActions()) {
            return false;
        }

        if ($canonicalUrl = $this->getCanonicalUrl()) {
            $this->addLinkCanonical($canonicalUrl);
        }
    }

    /**
     * @return $this|mixed|string
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     * @SuppressWarnings(PHPMD.NPathComplexity) 
     * @SuppressWarnings(PHPMD.ExcessiveMethodLength)
     */
    public function getCanonicalUrl()
    {
        if (!$this->config->isAddCanonicalUrl() || $this->isIgnoredCanonical()) {
            return false;
        }

        if ($canonicalRewrite = $this->getCanonicalRewrite()) {
            return $canonicalRewrite;
        }

        $productActions = [
            'catalog_product_view',
            'review_product_list',
            'review_product_view',
            'productquestions_show_index',
        ];

        $productCanonicalStoreId = false;
        $useCrossDomain          = true;

        if (in_array($this->fullAction, $productActions)) {
            $product = $this->registry->registry('current_product');

            if (!$product) {
                return;
            }

            $currentProductId    = $product->getId();
            $associatedProductId = $this->getAssociatedProductId($product);
            $productId           = ($associatedProductId) ?: $product->getId();

            $productCanonicalStoreId       = $product->getSeoCanonicalStoreId(); //canonical store id for current product
            $canonicalUrlForCurrentProduct = trim($product->getSeoCanonicalUrl() ?? '');

            $collection = $this->productCollectionFactory->create()
                ->addFieldToFilter('entity_id', $productId)
                ->addStoreFilter()
                ->addUrlRewrite();

            $collection->setFlag('has_stock_status_filter');

            $product      = $collection->getFirstItem();
            $canonicalUrl = $product->getProductUrl();

            if ($this->config->isAddLongestCanonicalProductUrl()
                && $this->config->isProductLongUrlEnabled($this->storeManager->getStore()->getId())
            ) {
                $canonicalUrl = $this->getLongestProductUrl($product, $canonicalUrl);
            }

            if ($canonicalUrlForCurrentProduct) {
                if (strpos($canonicalUrlForCurrentProduct, 'http://') !== false
                    || strpos($canonicalUrlForCurrentProduct, 'https://') !== false
                ) {
                    $canonicalUrl   = $canonicalUrlForCurrentProduct;
                    $useCrossDomain = false;
                } else {
                    $canonicalUrlForCurrentProduct = (substr($canonicalUrlForCurrentProduct, 0, 1) === '/')
                        ? substr($canonicalUrlForCurrentProduct, 1)
                        : $canonicalUrlForCurrentProduct;
                    $canonicalUrl                  = $this->context->getUrlBuilder()->getBaseUrl() . $canonicalUrlForCurrentProduct;
                }
            }
            $productLoaded = $this->productRepository->getById(
                $currentProductId,
                false,
                $this->storeManager->getStore()->getId()
            );

            //use custom canonical from products
            if ($productLoaded->getMSeoCanonical()) {
                $canonicalUrl = trim($productLoaded->getMSeoCanonical() ?? '');
                if (strpos($canonicalUrl, '://') === false) {
                    $canonicalUrl = $this->storeManager->getStore()->getBaseUrl() . ltrim($canonicalUrl ?? '', '/');
                }
            }
        } elseif ($this->fullAction === 'catalog_category_view') {
            $category = $this->registry->registry('current_category');
            if (!$category) {
                return;
            }
            $canonicalUrl = $category->getUrl();
        } else {
            $canonicalUrl              = $this->seoData->getBaseUri();
            $preparedCanonicalUrlParam = ($this->config->isAddStoreCodeToUrlsEnabled()
                && $this->stateService->isHomePage()) ? '' : ltrim($canonicalUrl ?? '', '/');
            $canonicalUrl              = $this->context->getUrlBuilder()->getUrl('',
                ['_direct' => $preparedCanonicalUrlParam]);
            $canonicalUrl              = strtok($canonicalUrl, '?');
        }

        if ($this->config->getCanonicalStoreWithoutStoreCode($this->storeManager->getStore()->getId())) {
            $storeCode    = $this->storeManager->getStore()->getCode();
            $canonicalUrl = str_replace('/' . $storeCode . '/', '/', $canonicalUrl);
            //setup crossdomian URL if this option is enabled
        } elseif ((($crossDomainStore = $this->config->getCrossDomainStore($this->storeManager->getStore()->getId()))
                || $productCanonicalStoreId)
            && $useCrossDomain) {
            if ($productCanonicalStoreId) {
                $crossDomainStore = $productCanonicalStoreId;
            }
            $mainBaseUrl    = $this->storeManager->getStore($crossDomainStore)->getBaseUrl();
            $currentBaseUrl = $this->storeManager->getStore()->getBaseUrl();
            $canonicalUrl   = str_replace($currentBaseUrl, $mainBaseUrl, $canonicalUrl);

            $mainSecureBaseUrl = $this->storeManager->getStore($crossDomainStore)
                ->getBaseUrl(UrlInterface::URL_TYPE_WEB, true);

            if ($this->storeManager->getStore()->isCurrentlySecure()
                || ($this->config->isPreferCrossDomainHttps()
                    && strpos($mainSecureBaseUrl, 'https://') !== false)) {
                $canonicalUrl = str_replace('http://', 'https://', $canonicalUrl);
            }
        }

        $canonicalUrl = $this->urlPrepare->deleteDoubleSlash($canonicalUrl);

        $page = (int)$this->request->getParam('page');
        if ($page > 1 && $this->config->isPaginatedCanonical()) {
            $canonicalUrl .= "?page=$page";
        }
        
        $canonicalUrl = $this->getPreparedTrailingCanonical($canonicalUrl);

        return $canonicalUrl;
    }

    /**
     * Check if canonical is ignored.
     * @return bool
     */
    public function isIgnoredCanonical()
    {
        foreach ($this->config->getCanonicalUrlIgnorePages() as $page) {
            if ($this->seoData->checkPattern($this->fullAction, $page)
                || $this->seoData->checkPattern($this->seoData->getBaseUri(), $page)) {
                return true;
            }
        }

        return false;
    }

    /**
     * @return string|bool
     */
    public function getCanonicalRewrite()
    {
        if ($canonicalRewriteRule = $this->canonicalRewriteService->getCanonicalRewriteRule()) {
            return $canonicalRewriteRule->getData('canonical');
        }

        return false;
    }

    /**
     * Get associated product Id
     *
     * @param Product $product
     *
     * @return bool|int
     */
    protected function getAssociatedProductId($product)
    {
        if ($product->getTypeId() !== Type::TYPE_SIMPLE) {
            return false;
        }

        $associatedProductId = false;

        if ($this->config->getAssociatedCanonicalConfigurableProduct()
            && ($parentConfigurableProductIds = $this
                ->productTypeConfigurable
                ->getParentIdsByChild($product->getId())
            )
            && isset($parentConfigurableProductIds[0])
            && $this->isProductEnabled($parentConfigurableProductIds[0])) {
            $associatedProductId = $parentConfigurableProductIds[0];
        }

        if (!$associatedProductId && $this->config->getAssociatedCanonicalGroupedProduct()
            && ($parentGroupedProductIds = $this
                ->productTypeGrouped
                ->getParentIdsByChild($product->getId())
            )
            && isset($parentGroupedProductIds[0])
            && $this->isProductEnabled($parentGroupedProductIds[0])) {
            $associatedProductId = $parentGroupedProductIds[0];
        }

        if (!$associatedProductId && $this->config->getAssociatedCanonicalBundleProduct()
            && ($parentBundleProductIds = $this
                ->productTypeBundle
                ->getParentIdsByChild($product->getId())
            )
            && isset($parentBundleProductIds[0])
            && $this->isProductEnabled($parentBundleProductIds[0])) {
            $associatedProductId = $parentBundleProductIds[0];
        }

        return $associatedProductId;
    }

    /**
     * return bool
     *
     * @param string $id
     *
     * @return bool
     * @return bool
     * @throws NoSuchEntityException
     */
    protected function isProductEnabled($id)
    {
        $product = $this->productRepository->getById(
            $id,
            false,
            $this->storeManager->getStore()->getId()
        );

        return $product->getStatus() === Status::STATUS_ENABLED;
    }

    /**
     * Get longest product url
     *
     * @param object $product
     * @param string $canonicalUrl
     *
     * @return string
     */
    protected function getLongestProductUrl($product, $canonicalUrl)
    {
        $rewriteData = $this->urlRewrite->addFieldToFilter('entity_type', 'product')
            ->addFieldToFilter('redirect_type', 0)
            ->addFieldToFilter('store_id', $this->storeManager->getStore()->getId())
            ->addFieldToFilter('entity_id', $product->getId());

        if ($rewriteData && $rewriteData->getSize() > 1) {
            $urlPath = [];
            foreach ($rewriteData as $rewrite) {
                $requestPath             = $rewrite->getRequestPath();
                $requestPathExploded     = explode('/', $requestPath);
                $categoryCount           = count($requestPathExploded);
                $urlPath[$categoryCount] = $requestPath;
            }

            if ($urlPath) {
                $canonicalUrl = $this->storeManager->getStore()->getBaseUrl() . $urlPath[max(array_keys($urlPath))];
            }
        }

        return $canonicalUrl;
    }

    /**
     * Get Canonical with prepared Trailing slash (depending on Trailing slash config)
     *
     * @param string $canonicalUrl
     *
     * @return string
     */
    protected function getPreparedTrailingCanonical($canonicalUrl)
    {
        if ($this->config->getTrailingSlash() === Config::TRAILING_SLASH
            && substr($canonicalUrl, -1) !== '/'
            && strpos($canonicalUrl, '?') === false) {
            $canonicalUrl .= '/';
        } elseif ($this->config->getTrailingSlash() === Config::NO_TRAILING_SLASH
            && substr($canonicalUrl, -1) === '/') {
            if ($this->checkHomePageCanonical($canonicalUrl)) {
                return $canonicalUrl;
            }

            $canonicalUrl = substr($canonicalUrl, 0, -1);
        }

        return $canonicalUrl;
    }

    /**
     * @param string $canonicalUrl
     *
     * @return bool
     */

    protected function checkHomePageCanonical($canonicalUrl)
    {
        if ($this->stateService->isHomePage()
            && $this->config->isAddStoreCodeToUrlsEnabled()
            && $this->storeManager->getStore()->getBaseUrl(UrlInterface::URL_TYPE_WEB)
            . $this->storeManager->getStore()->getCode()
            . '/' === $this->context->getUrlBuilder()->getCurrentUrl()
            && $this->context->getUrlBuilder()->getCurrentUrl() === $canonicalUrl) {
            return true;
        }

        return false;

    }

    /**
     * Create canonical.
     *
     * @param string $canonicalUrl
     *
     * @return void
     */
    public function addLinkCanonical($canonicalUrl)
    {
        $pageConfig = $this->context->getPageConfig();
        $type       = 'canonical';
        $pageConfig->addRemotePageAsset(
            htmlentities($canonicalUrl),
            $type,
            ['attributes' => ['rel' => $type]]
        );
    }
}
