<?php

namespace Overdose\RelatedCategories\Block;

use Magento\Catalog\Block\Product\View\Description;

/**
 * Class Categories
 * @package Overdose\RelatedCategories\Block
 */
class Categories extends Description
{
    /**
     * @var \Magento\Catalog\Model\CategoryRepository
     */
    protected $categoryRepository;

    /**
     * @var \Magento\Framework\App\ResourceConnection
     */
    private $resourceConnection;

    /**
     * @var \Magento\Catalog\Model\ResourceModel\Category\CollectionFactory
     */
    private $categoryCollectionFactory;

    /**
     * Categories constructor.
     * @param \Magento\Framework\App\ResourceConnection $resourceConnection
     * @param \Magento\Catalog\Model\CategoryRepository $categoryRepository
     * @param \Magento\Framework\View\Element\Template\Context $context
     * @param \Magento\Framework\Registry $registry
     * @param array $data
     */
    public function __construct(
        \Magento\Framework\App\ResourceConnection $resourceConnection,
        \Magento\Catalog\Model\CategoryRepository $categoryRepository,
        \Magento\Catalog\Model\ResourceModel\Category\CollectionFactory $categoryCollectionFactory,
        \Magento\Framework\View\Element\Template\Context $context,
        \Magento\Framework\Registry $registry,
        array $data = []
    ) {
        parent::__construct($context, $registry, $data);
        $this->categoryRepository = $categoryRepository;
        $this->resourceConnection = $resourceConnection;
        $this->categoryCollectionFactory = $categoryCollectionFactory;
    }

    /**
     * @param int $id
     * @return \Magento\Catalog\Api\Data\CategoryInterface|mixed|null
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getCategoryById($id)
    {
        return $this->categoryRepository->get($id, $this->_storeManager->getStore()->getId());
    }

    /**
     * @param int $chunkSize
     * @return string
     * @throws \Magento\Framework\Exception\LocalizedException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getCrumbs($chunkSize = 0)
    {
        $evercrumbs = [];
        $sortingRang = '';
        $categoryCollection = $this->getCategoryCollection();

        foreach ($categoryCollection as &$categoryChild) {
            $breadcrumbCategories = $categoryChild->getParentCategories();
            if ($breadcrumbCategories && array_values($breadcrumbCategories)[0]->getLevel() == 2) {
                foreach ($breadcrumbCategories as $category) {
                    $evercrumbs[$categoryChild->getId()][$category->getLevel()] = array(
                        'category_id' => $category->getId(),
                        'label' => $category->getName(),
                        'title' => $category->getName(),
                        'link' => $category->getUrl(),
                        'key' => $category->getUrlKey()
                    );
                    $sortingRang .= str_replace(' ', '', strtolower($category->getName()));
                }

                if (count($categoryChild->getParentCategories())
                    && array_key_exists('2', $evercrumbs[$categoryChild->getId()])
                ) {
                    $evercrumbs[$categoryChild->getId()][2]['rang'] =
                        $evercrumbs[$categoryChild->getId()][2]['label'] . $sortingRang;
                    $sortingRang = '';
                }

                if (isset($evercrumbs[$categoryChild->getId()])) {
                    ksort($evercrumbs[$categoryChild->getId()]);
                }
            }
        }

        if (count($evercrumbs) < 1) {
            return '';
        }

        $evercrumbs = $this->sortEverCrumbs($evercrumbs, false);

        $html = '';

        if ($chunkSize > 0) {
            $chunks = array_chunk($evercrumbs, $chunkSize, true);
            foreach ($chunks as $chunk) {
                $html .= $this->createCategoryCrumb($chunk, false);
            }
        } else {
            $html .= $this->createCategoryCrumb($evercrumbs, false);
        }

        return $html;
    }

    /**
     * Sort related categories by rang (generated by cat names and some magic)
     * @param $evercrumbs
     * @return array
     */
    private function sortEverCrumbs($evercrumbs, $tree = true)
    {
        if ($tree) {
            usort($evercrumbs, function ($a, $b) {
                return $a[2]['rang'] <=> $b[2]['rang'];
            });
        } else {
            usort($evercrumbs, function ($a, $b) {
                return last($a)['title'] <=> last($b)['title'];
            });
        }


        return $evercrumbs;
    }

    /**
     * @param $evercrumbs
     * @param bool $tree is need to build categories tree (true) or just show lower level (false)
     * @return string
     */
    private function createCategoryCrumb($evercrumbs, $tree = true)
    {
        $html = '<dl class="category-crumb">';
        foreach ($evercrumbs as $everCrumb) {
            $htmlCrumb = '';
            $crumbCount = count($everCrumb);
            $counter = 0;

            if ($tree) {
                foreach ($everCrumb as $crumb) {
                    $counter++;
                    $htmlCrumb .= '<a href="' . $crumb['link'] . '"> ' . $crumb['title'] . ' </a>';
                    if ($counter === $crumbCount) {
                        break;
                    } else {
                        $htmlCrumb .= '>';
                    }
                }
            } else {
                $crumb = last($everCrumb);
                $htmlCrumb .= '<a href="' . $crumb['link'] . '"> ' . $crumb['title'] . ' </a>';
            }

            $html .= '<li>' . $htmlCrumb . '</li>';
        }
        $html .= '</dl>';
        return $html;
    }

    /**
     * @return \Magento\Catalog\Model\ResourceModel\Category\Collection|\Magento\Framework\Data\Collection
     * @throws \Magento\Framework\Exception\LocalizedException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getCategoryCollection()
    {
        $product = $this->getProduct();

        $collection = $this->resourceConnection->getConnection();
        if ($collection->isTableExists($collection->getTableName('smile_virtualcategory_catalog_category_product_position'))) {
            $select = $collection->select()
                ->from(
                    ['main_table' => 'smile_virtualcategory_catalog_category_product_position'],
                    ['category_id']
                )
                ->where('product_id = ?', $product->getId())
                ->where('store_id = ?', $this->_storeManager->getStore()->getId());
            $virtualCategoryData = $collection->fetchCol($select);

            if (count($virtualCategoryData) > 0) {
                $categoryCollection = $this->categoryCollectionFactory->create();
                $categoryCollection->addAttributeToSelect('name');
                $categoryCollection->addAttributeToSort('level', $categoryCollection::SORT_ORDER_DESC);
                $categoryCollection->addAttributeToFilter('entity_id', ['in' => $virtualCategoryData]);
            } else {
                $categoryCollection = $product->getCategoryCollection();
                $categoryCollection->addAttributeToSelect('name');
                $categoryCollection->addAttributeToSort('level', $categoryCollection::SORT_ORDER_DESC)
                    ->addAttributeToFilter('path',
                        ['like' => '1/' . $this->_storeManager->getStore()->getRootCategoryId() . '/%']);
            }
        } else {
            $categoryCollection = $product->getCategoryCollection();
            $categoryCollection->addAttributeToSelect('name');
            $categoryCollection->addAttributeToSort('level', $categoryCollection::SORT_ORDER_DESC)
                ->addAttributeToFilter('path',
                    ['like' => '1/' . $this->_storeManager->getStore()->getRootCategoryId() . '/%']);
        }
        $categoryCollection
            ->addAttributeToFilter('name', ['nin' => ['Shop', 'Sale']]);
        return $categoryCollection;
    }
}
