<?php

namespace Overdose\InventoryShipping\Model\SourceSelection;

use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\InventoryApi\Api\Data\SourceItemInterface;
use Magento\InventoryApi\Api\SourceItemRepositoryInterface;
use Magento\InventorySourceSelectionApi\Model\GetInStockSourceItemsBySkusAndSortedSource as DefaultGetInStockSourceItemsBySkusAndSortedSource;

/**
 * Retrieve source items for a defined set of skus and sorted source codes
 *
 * Useful for determining presence in stock and source selection
 *
 * @api
 */
class GetInStockSourceItemsBySkusAndSortedSource extends DefaultGetInStockSourceItemsBySkusAndSortedSource
{
    /**
     * @var SourceItemRepositoryInterface
     */
    private $sourceItemRepository;

    /**
     * @var SearchCriteriaBuilder
     */
    private $searchCriteriaBuilder;

    /**
     * @param SourceItemRepositoryInterface $sourceItemRepository
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @SuppressWarnings(PHPMD.LongVariable)
     */
    public function __construct(
        SourceItemRepositoryInterface $sourceItemRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder
    ) {
        $this->sourceItemRepository = $sourceItemRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        parent::__construct($sourceItemRepository, $searchCriteriaBuilder);
    }

    /**
     * Retrieve source items for a defined set of SKUs and sorted source codes
     *
     * Override core method to add quantity condition to handle negative source quantities
     *
     * @param array $skus
     * @param array $sortedSourceCodes
     * @return SourceItemInterface[]
     */
    public function execute(array $skus, array $sortedSourceCodes): array
    {
        $skus = array_map('strval', $skus);
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter(SourceItemInterface::SKU, $skus, 'in')
            ->addFilter(SourceItemInterface::SOURCE_CODE, $sortedSourceCodes, 'in')
            ->addFilter(SourceItemInterface::STATUS, SourceItemInterface::STATUS_IN_STOCK)
            ->addFilter(SourceItemInterface::QUANTITY, 0, 'gt')
            ->create();

        $items = $this->sourceItemRepository->getList($searchCriteria)->getItems();

        $itemsSorting = [];
        foreach ($items as $item) {
            $itemsSorting[] = array_search($item->getSourceCode(), $sortedSourceCodes, true);
        }

        array_multisort($itemsSorting, SORT_NUMERIC, SORT_ASC, $items);
        return $items;
    }
}
