<?php
namespace Overdose\DataLayer\Plugin;

use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\DataObject;
use Magento\Store\Model\ScopeInterface;
use Magento\Framework\App\Response\Http;
use Magento\Framework\App\ResponseInterface;
use Magento\Framework\View\Result\Layout;
use Magento\Framework\Event\ManagerInterface as EventManager;

class JsFooterPlugin
{
    const XML_PATH_DEV_MOVE_JS_TO_BOTTOM = 'dev/js/move_script_to_bottom';

    /**
     * @var ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * @var EventManager
     */
    protected $eventManager;

    /**
     * Constructor
     *
     * @param ScopeConfigInterface $scopeConfig
     * @param EventManager $eventManager
     */
    public function __construct
    (
        ScopeConfigInterface $scopeConfig,
        EventManager $eventManager
    ) {
        $this->scopeConfig = $scopeConfig;
        $this->eventManager = $eventManager;
    }

    /**
     * Around after render
     *
     * @param \Magento\Theme\Controller\Result\JsFooterPlugin $plugin
     * @param callable $proceed
     * @param Layout $subject
     * @param Layout $result
     * @param ResponseInterface $httpResponse
     * @return Layout
     */
    public function aroundAfterRenderResult(
        \Magento\Theme\Controller\Result\JsFooterPlugin $plugin,
        callable $proceed,
        Layout $subject,
        Layout $result,
        ResponseInterface 
        $httpResponse
    ) {
        if (!$this->isActive()) {
            return $proceed($subject, $result, $httpResponse);
        }

        if ($this->isDeferEnabled()) {
            $content = (string) $httpResponse->getContent();

            $httpResponse->setContent
            (
                $this->transform($content)
            );
        }

        /**
         * Create transport object
         *
         * @var DataObject $transport
         */
        $transport = new DataObject
        (
            [
                'result'    => $result,
                'response'  => $httpResponse
            ]
        );

        /**
         * Notify others for schema
         */
        $this->eventManager->dispatch('footer_after_get_result', ['transport' => $transport]);

        return $transport->getResult();
    }

    /**
     * Around before
     *
     * @param \Magento\Theme\Controller\Result\JsFooterPlugin $plugin
     * @param callable $proceed
     * @param Http $subject
     */
    public function aroundBeforeSendResponse(
        \Magento\Theme\Controller\Result\JsFooterPlugin $plugin,
        callable $proceed,
        Http $subject
    ) {
        if (!$this->isActive()) {
            return $proceed($subject);
        }

        /**
         * Response content
         *
         * @var string $content
         */
        $content = $subject->getContent();

        $subject->setContent
        (
            $this->transform($content)
        );
    }

    /**
     * Transform content
     *
     * @param string $content
     * @return string
     */
    private function transform($content)
    {
        if (!$content)
        {
            return $content;
        }

        if (strpos($content, '</body') !== false)
        {
            if ($this->isDeferEnabled())
            {
                /* Start - Custom codes */
                $scripts = $this->extractScriptTags($content);
                $content = str_replace('</body', $scripts . "\n</body", $content);
                /* End - Custom codes */
            }
        }

        return $content;
    }

    /**
     * Extracts and returns script tags found in given content.
     *
     * @param string $content
     */
    private function extractScriptTags(&$content): string
    {
        $skip = [
            'data-ommit',
            'www.googletagmanager.com/gtm.js',
            'ec.js',
            'ec.min.js',
            'ec4.js',
            'ec4.min.js'
        ];
        $scripts = '';
        $scriptOpen = '<script';
        $scriptClose = '</script>';
        $scriptOpenPos = strpos($content, $scriptOpen);
        $newContent = $content;

        while ($scriptOpenPos !== false) {
            $scriptClosePos = strpos($newContent, $scriptClose, $scriptOpenPos);
            $script = substr($newContent, $scriptOpenPos, $scriptClosePos - $scriptOpenPos + strlen($scriptClose));
            $isXMagentoTemplate = strpos($script, 'text/x-magento-template') !== false;

            if ($isXMagentoTemplate) {
                $scriptOpenPos = strpos($newContent, $scriptOpen, $scriptClosePos);
                continue;
            }
            $isRepalce = false;
            foreach ($skip as $value)
            {
                /**
                 * Prevent script from being moved to bottom
                 */
                if (false !== strpos($script, $value))
                {
                    $isRepalce = true;
                }
            }
            if (!$isRepalce) {
                $scripts .= "\n" . $script;
                $content = str_replace($script, '', $content);
            }
            $newContent = str_replace($script, '', $newContent);

            // Script cut out, continue search from its position.
            $scriptOpenPos = strpos($newContent, $scriptOpen, $scriptOpenPos);
        }
        
        return $scripts;
    }

    /**
     * Returns information whether moving JS to footer is enabled
     *
     * @return bool
     */
    private function isDeferEnabled(): bool
    {
        return $this->scopeConfig->isSetFlag(static::XML_PATH_DEV_MOVE_JS_TO_BOTTOM,ScopeInterface::SCOPE_STORE);
    }

    /**
     * Returns information whether moving JS to footer is enabled
     *
     * @return bool
     */
    private function isActive(): bool
    {
        return $this->scopeConfig->isSetFlag('ec/general/active',ScopeInterface::SCOPE_STORE);
    }
}
