<?php

declare(strict_types=1);

namespace Overdose\AmStoreLocatorExport\Model\Export;

use Magento\Framework\App\Filesystem\DirectoryList;
use Magento\Framework\Filesystem\Driver\File;
use Magento\Framework\Filesystem;
use Psr\Log\LoggerInterface;
use Amasty\Storelocator\Model\LocationFactory as AmastyLocationModel;

/**
 * Generate amasty locations and save in locations_export.csv file.
 */
class GenerateLocationsFile implements \Overdose\AmStoreLocatorExport\Api\Data\GenerateAmastyLocationsExport
{
    /** @var Filesystem\Directory\WriteInterface */
    private $directory;

    /** @var LoggerInterface */
    private $logger;

    /** @var DirectoryList */
    private $directoryList;

    /** @var File */
    private $file;

    /** @var AmastyLocationModel */
    private $amastyLocationModel;

    /**
     * @param Filesystem $filesystem
     * @param LoggerInterface $logger
     * @param DirectoryList $directoryList
     * @param File $file
     * @param AmastyLocationModel $amastyLocationModel
     * @throws \Magento\Framework\Exception\FileSystemException
     */
    public function __construct(
        Filesystem $filesystem,
        LoggerInterface $logger,
        DirectoryList $directoryList,
        File $file,
        AmastyLocationModel $amastyLocationModel
    ) {
        $this->directory = $filesystem->getDirectoryWrite(DirectoryList::MEDIA);
        $this->logger = $logger;
        $this->directoryList = $directoryList;
        $this->file = $file;
        $this->amastyLocationModel = $amastyLocationModel;
    }

    /**
     * @return string
     * @throws \Magento\Framework\Exception\FileSystemException
     */
    public function generateAmastyLocationsExportFile(): string
    {
        $filePath = 'amasty/amlocator/export/locations_export.csv';
        $file = $this->directoryList->getPath(DirectoryList::MEDIA) . DIRECTORY_SEPARATOR . $filePath;
        $fileContent = '';

        try {
            if ($this->file->isExists($file)) {
                $this->file->deleteFile($file);
            }

            $this->directory->create('amasty/amlocator/export');
            $stream = $this->directory->openFile($filePath, 'w+');
            $stream->lock();
            $headers = $this->getHeaderForCsv();
            $stream->writeCsv($headers);

            $locationsCollection = $this->amastyLocationModel->create()->getCollection();
            $locationsCollection->getSelect()->joinLeft(
                ['a_a_g' => \Amasty\Storelocator\Setup\Operation\CreateGalleryTable::TABLE_NAME],
                'a_a_g.location_id = main_table.id',
                ['image_name']
            );
            $locationsCollection->addFieldToSelect(
                [
                    'id',
                    'name',
                    'country',
                    'city',
                    'zip',
                    'address',
                    'state',
                    'lat',
                    'lng',
                    'description',
                    'phone',
                    'email',
                    'website',
                    'status',
                    'stores',
                    'schedule',
                    'url_key',
                    'meta_title',
                    'meta_description'
                ]
            );

            /** @var \Amasty\Storelocator\Model\Location $location */
            foreach ($locationsCollection as $location) {
                $locationData = array_values($location->getData());
                $stream->writeCsv($locationData);
            }

            $fileContent = $this->file->fileGetContents($file);
            $this->file->deleteFile($file);
        } catch (\Exception $exception) {
            $this->logger->error(
                __('OD: Something went wrong while generating amasty locations export file: ' . $exception->getMessage())
            );
        }
        return $fileContent;
    }

    /**
     * @return string[]
     */
    private function getHeaderForCsv(): array {
        return [
            'Location ID',
            'Name',
            'Country',
            'City',
            'Zip',
            'Address',
            'State',
            'Latitude',
            'Longitude',
            'Description',
            'Phone',
            'Email',
            'Website',
            'Status',
            'Stores',
            'Schedule',
            'Url Key',
            'Meta Title',
            'Meta Description',
            'Image Name'
        ];
    }
}
