<?php

namespace BrittainWynyard\OneAsics\OAuth2\Client\Provider;

use GuzzleHttp\Exception\BadResponseException;
use League\OAuth2\Client\Provider\AbstractProvider;
use BrittainWynyard\OneAsics\Helper\Config;
use League\OAuth2\Client\Provider\Exception\IdentityProviderException;
use League\OAuth2\Client\Token\AccessToken;
use League\OAuth2\Client\Token\AccessTokenInterface;
use League\OAuth2\Client\Tool\ArrayAccessorTrait;
use League\OAuth2\Client\Tool\BearerAuthorizationTrait;
use Psr\Http\Message\RequestInterface;
use Psr\Http\Message\ResponseInterface;

/**
 * Class OneAsicsEmail
 */
class OneAsicsEmail extends AbstractProvider
{
    use ArrayAccessorTrait,
        BearerAuthorizationTrait;

    /**
     * @var string
     */
    private $responseError = 'error';

    /**
     * @var string
     */
    private $responseCode;

    private $configHelper;

    /**
     * OneAsicsEmail constructor.
     * @param \BrittainWynyard\OneAsics\Helper\Config $helper
     * @param \Psr\Log\LoggerInterface $logger
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function __construct(
        \BrittainWynyard\OneAsics\Helper\Config $helper,
        \Psr\Log\LoggerInterface $logger
    ) {
        $this->logger = $logger;
        $this->configHelper = $helper;

        $options = [
            'clientId'     => $this->configHelper->getClientId(),    // The client ID assigned to you by the provider
            'clientSecret' => $this->configHelper->getClientSecret(),   // The client password assigned to you by the provider
            'redirectUri'  =>  $this->configHelper->getRedirectUri('oneasics/change/email')
        ];

        parent::__construct($options);
    }

    /**
     * Get authorization url to begin OAuth flow
     *
     * @return string
     */
    public function getBaseAuthorizationUrl()
    {
        return $this->configHelper->getBaseUrl() . $this->configHelper::EMAIL_API_END_POINT;
    }

    /**
     * Get access token url to retrieve token
     *
     * @return string
     */
    public function getBaseAccessTokenUrl(array $params)
    {
        return $this->configHelper->getBaseUrl() . $this->configHelper::ACCESS_TOKEN_API_END_POINT;
    }

    /**
     * Get provider url to fetch user details
     *
     * @param  AccessToken $token
     *
     * @return string
     */
    public function getResourceOwnerDetailsUrl(AccessToken $token)
    {
        return $this->configHelper->getBaseUrl() . $this->configHelper::CHANGE_EMAIL_API_END_POINT;
    }

    /**
     * Get the default scopes used by this provider.
     *
     * This should not be a complete list of all scopes, but the minimum
     * required for the provider user interface!
     *
     * @return array
     */
    protected function getDefaultScopes()
    {
        return [];
    }

    /**
     * Returns the string that should be used to separate scopes when building
     * the URL for requesting an access token.
     *
     * @return string Scope separator, defaults to ' '
     */
    protected function getScopeSeparator()
    {
        return ' ';
    }

    /**
     * @inheritdoc
     */
    protected function checkResponse(ResponseInterface $response, $data)
    {
        if (!empty($data[$this->responseError])) {
            $error = $data[$this->responseError];
            if (!is_string($error)) {
                $error = var_export($error, true);
            }
            $code  = $this->responseCode && !empty($data[$this->responseCode])? $data[$this->responseCode] : 0;
            if (!is_int($code)) {
                $code = intval($code);
            }

            throw new IdentityProviderException($error, $code, $data);
        }
    }

    /**
     * Generate a user object from a successful user details request.
     *
     * @param array $response
     * @param AccessToken $token
     * @return array
     */
    protected function createResourceOwner(array $response, AccessToken $token)
    {
        return $response;
    }

    /**
     * Sends a request and returns the parsed response.
     *
     * @param  RequestInterface $request
     * @throws IdentityProviderException
     * @return mixed
     */
    public function getParsedResponse(RequestInterface $request)
    {
        try {
            $response = $this->getResponse($request);
        } catch (BadResponseException $e) {
            $response = $e->getResponse();
        }
        $parsed = $this->parseResponse($response);

        return $parsed;
    }

    /**
     * Returns a prepared request for requesting an access token.
     *
     * @param array $params Query string parameters
     * @return RequestInterface
     */
    protected function getAccessTokenRequest(array $params)
    {
        $method  = $this->getAccessTokenMethod();
        $url     = $this->getAccessTokenUrl($params);

        $options = $this->optionProvider->getAccessTokenOptions($this->getAccessTokenMethod(), $params);

        if ($this->configHelper->isDebugMode()) {
            $this->logger->info($url);
        }

        if ($this->configHelper->isDebugMode()) {
            $this->logger->info(print_r($options, true));
        }

        return $this->getRequest($method, $url, $options);
    }

    /**
     * Builds the authorization URL.
     *
     * @param  array $options
     * @return string Authorization URL
     */
    public function getAuthorizationUrl(array $options = [])
    {
        $defaultOptions = [
            'redirect_uri'            => $this->configHelper->getRedirectUri('oneasics/change/email'),
            'style'                   => $this->configHelper->getStyle(),
            'grant_type'              => 'code',
            'privacy_url'             => $this->configHelper->getPrivacyUrl(),
            'terms_url'               => $this->configHelper->getTermsUrl(),
            'terms_privacy_country'   => $this->configHelper->getTermsPrivacyCountry(),
            'terms_privacy_version'   => $this->configHelper->getTermsPrivacyVersion(),
            'locale'                  => $this->configHelper->getLocale(),
        ];
        $options = array_merge($defaultOptions, $options);
        return parent::getAuthorizationUrl($options);
    }

    /**
     * Requests an access token using a specified grant and option set.
     *
     * @param  mixed $grant
     * @param  array $options
     * @throws IdentityProviderException
     * @return AccessTokenInterface
     */
    public function getAccessToken($grant, array $options = [])
    {
        $grant = $this->verifyGrant($grant);

        $params = [
            'client_id'     => $this->clientId,
            'client_secret' => $this->clientSecret,
            'redirect_uri'  => $this->redirectUri
        ];

        $params   = $grant->prepareRequestParameters($params, $options);
        $request  = $this->getAccessTokenRequest($params);

        if ($this->configHelper->isDebugMode()) {
            $this->logger->info(print_r($params, 1));
        }

        $response = $this->getParsedResponse($request);
        if (false === is_array($response)) {

            throw new UnexpectedValueException(
                'Invalid response received from Authorization Server. Expected JSON.'
            );
        }
        $prepared = $this->prepareAccessTokenResponse($response);

        $token    = $this->createAccessToken($prepared, $grant);

        return $token;
    }
}
