# BrittainWynyard_GiftcardAccount

## Overview

This module extends the core Magento_GiftCardAccount functionality to provide Brittain Wynyard specific giftcard functionality. Currently that consists of two key features:

  - Synchronization of Magento giftcard accounts with Accredo giftcard balances.
  - The ability to hide the checkout giftcard inputs through configuration

## Key Features

### 🔄 **Automatic Accredo Synchronization**
- Real-time sync with Accredo system when gift cards are requested
- Automatic balance updates when discrepancies are detected
- Creation of new gift cards from Accredo data
- Comprehensive error handling and logging

### 🔌 **Plugin Architecture**
- Intercepts `Magento\GiftCardAccount\Model\Manager::requestByCode()` calls
- Performs sync **before** Magento retrieves gift card from database
- Ensures users always get the most up-to-date balance from Accredo


## Architecture

### Components

```
BrittainWynyard/GiftcardAccount/
├── Api/
│   └── AccredoProcessorInterface.php          # Service contract
├── Helper/
│   └── Config.php                             # Configuration helper
├── Model/
│   ├── Accredo/
│   │   └── Processor.php                      # Core sync implementation
│   └── History.php                            # Custom history model
├── Plugin/
│   └── Model/
│       └── GiftcardAccount/
│           └── ManagerPlugin.php              # Manager interception
└── etc/
    ├── adminhtml/
    │   └── system.xml                         # Admin configuration
    ├── config.xml                             # Default configuration values
    ├── di.xml                                 # Dependency injection config
    └── module.xml                             # Module configuration
```

### Sync Flow

1. **Gift Card Request** → Manager::requestByCode() is called
2. **Before Plugin** → AccredoProcessor::syncGiftcard() executes
3. **Accredo API Call** → Fetch latest gift card data from Accredo
4. **Business Logic** → Compare and sync data based on scenarios
5. **Database Update** → Update/create gift card in Magento
6. **Core Method** → Magento retrieves updated gift card from database
7. **After Plugin** → Log final results

## Business Logic

### Synchronization Scenarios

| Scenario | Magento | Accredo | Action |
|----------|---------|---------|--------|
| 1 | ✅ Exists | ✅ Exists + Same Balance | ⏭️ Do nothing |
| 2 | ✅ Exists | ✅ Exists + Different Balance | 🔄 Update Magento balance |
| 3 | ❌ Missing | ✅ Exists | ➕ Create new gift card in Magento |
| 4 | ✅ Exists | ❌ Missing | ⏭️ Skip (do nothing) |

### Data Mapping

When creating/updating gift cards, the following Accredo data is used:

- **code** → Gift card code (provided as parameter)
- **card_balance** → Current balance (from Accredo response, float)
- **status** → Enabled state (defaults to 1/enabled)
- **issuedate** → Converted to expiration date (Y-m-d format, validated to be future)
- **card_no** → Accredo card number (stored for reference)
- **cardid** → Accredo card ID (stored for reference)
- **ModifiedDate** → Last modification date in Accredo
- **ModifiedUserCode** → User who last modified in Accredo

**Note:** Recipient email and name are currently set to empty strings as they're not provided by the Accredo API response.

## Installation & Setup

### 1. Enable Module
```bash
php bin/magento module:enable BrittainWynyard_GiftcardAccount
php bin/magento setup:upgrade
php bin/magento cache:flush
```

### 2. Configuration
Navigate to `Stores > Configuration > Gift Cards > Accredo Integration` to configure:

**Configuration Storage:** Settings are stored using the path structure `bw_giftcard_accounts/accredo_integration/{setting}` in the `core_config_data` table while appearing in the dedicated Gift Cards configuration section.

#### Accredo API Configuration
- **Enable Accredo Integration**: Master switch for synchronization
- **API Endpoint URL**: Base URL for Accredo API (e.g., `https://hi9nn8xl27.execute-api.ap-southeast-2.amazonaws.com/`)
- **Bearer Token**: Authentication token (encrypted storage)
- **Request Timeout**: API timeout in seconds (default: 30)
- **Use Mock Data**: Enable for testing without real API calls

#### Synchronization Settings
- **Auto-Create Cards**: Create new Magento cards when found in Accredo
- **Auto-Update Balances**: Update Magento balances to match Accredo

The module automatically handles configuration validation and provides fallback to mock data if API calls fail.

### 3. Theme Integration

**⚠️ Required Theme CSS**

The module adds a CSS class `bw-giftcard-hidden` to the body element when gift cards should be hidden. You **must** add the following CSS rule to your theme's LESS files:

```less
// Hide gift card section when configuration is disabled
body.bw-giftcard-hidden {
    .payment-methods-container {
        .opc-payment-additional.giftcardaccount {
            display: none !important;
        }
    }
}
```

**Suggested Location:** Add this rule to your checkout-specific LESS file (e.g., `_payment-container.less` or similar).

**Testing:** 
- Set "Show Gift Card Input at Checkout" to "No" in admin configuration
- Visit checkout page - gift card input should be hidden
- Add HTTP header `X-FLAG-CHECKOUT-GIFTCARD: 1` - gift card input should become visible

## API Integration

### Accredo API Implementation
The module makes real HTTP requests to the Accredo gift card system:

**Request Format:**
```
POST https://your-accredo-endpoint.com/asics-gift-card
Authorization: Bearer <your-token>
Content-Type: text/plain

{
  "number": 94133345181
}
```

**Response Handling:**
- **200 OK**: Gift card found, data parsed and synchronized
- **404 Not Found**: Gift card doesn't exist in Accredo (skips sync)
- **Other errors**: Logged and sync operation fails gracefully



## Logging

### Log Locations
All sync activities are logged to Magento's standard log files:

- **var/log/system.log** - General sync information
- **var/log/exception.log** - Error details and stack traces

### Log Prefixes
- `BrittainWynyard_GiftcardAccount: requestByCode method called`
- `BrittainWynyard_GiftcardAccount: Starting Accredo sync for gift card`
- `BrittainWynyard_GiftcardAccount: Gift card balance updated successfully`
- `BrittainWynyard_GiftcardAccount: New gift card created successfully`
- `BrittainWynyard_GiftcardAccount: Gift card retrieved successfully with synced data`
- `BrittainWynyard_GiftcardAccount: Accredo sync completed in before plugin`
- `BrittainWynyard_GiftcardAccount: Making Accredo API call`
- `BrittainWynyard_GiftcardAccount: Using sample override response (mock mode enabled)`

### Log Details
Each log entry includes:
- Gift card code
- Balance information
- Website ID
- Sync results
- Error details (when applicable)

## Error Handling

### Graceful Degradation
- Accredo sync failures don't break gift card functionality
- Detailed error logging for troubleshooting
- Gift card requests continue with existing Magento data when sync fails

### Common Error Scenarios
- **Network timeouts** → Logged and skipped
- **Invalid Accredo data** → Validated and corrected
- **Database save failures** → Retry with existing card update
- **Duplicate codes** → Automatically handled

## Dependencies

### Required Modules
- **Magento_GiftCardAccount** - Core gift card functionality and interfaces
- **Magento_Store** - Store management and scope interfaces
- **Magento_Config** - Configuration management and admin interfaces

**Note:** These dependencies are defined in `etc/module.xml` to ensure proper initialization order. The module also uses services from `Magento_Framework` (dependency injection, HTTP client, logging, etc.) which are core framework dependencies.

## Support & Development

### Troubleshooting
1. **Check logs** → Review `var/log/system.log` for sync details
2. **Verify configuration** → Ensure module is enabled and configured
4. **Database verification** → Check `magento_giftcardaccount` table