<?php
/**
 * BrittainWynyard GiftcardAccount Manager Plugin
 *
 * @package BrittainWynyard_GiftcardAccount
 */

namespace BrittainWynyard\GiftcardAccount\Plugin\Model\GiftcardAccount;

use BrittainWynyard\GiftcardAccount\Api\AccredoProcessorInterface;
use Magento\GiftCardAccount\Api\Data\GiftCardAccountInterface;
use Magento\GiftCardAccount\Model\Giftcardaccount;
use Magento\GiftCardAccount\Model\Manager;
use Psr\Log\LoggerInterface;

/**
 * Plugin for Magento\GiftCardAccount\Model\Manager
 */
class ManagerPlugin
{
    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var AccredoProcessorInterface
     */
    private $accredoProcessor;

    /**
     * @param LoggerInterface $logger
     * @param AccredoProcessorInterface $accredoProcessor
     */
    public function __construct(
        LoggerInterface $logger,
        AccredoProcessorInterface $accredoProcessor
    ) {
        $this->logger = $logger;
        $this->accredoProcessor = $accredoProcessor;
    }

    /**
     * Before plugin for requestByCode method
     *
     * @param Manager $subject
     * @param string $code
     * @param int|null $websiteId
     * @param float|null $balanceGTE
     * @param bool $onlyEnabled
     * @param bool $notExpired
     * @return array
     */
    public function beforeRequestByCode(
        Manager $subject,
        string $code,
        ?int $websiteId = null,
        ?float $balanceGTE = null,
        bool $onlyEnabled = true,
        bool $notExpired = true
    ): array {
        $this->logger->info('BrittainWynyard_GiftcardAccount: requestByCode method called', [
            'code' => $code,
            'websiteId' => $websiteId,
            'balanceGTE' => $balanceGTE,
            'onlyEnabled' => $onlyEnabled,
            'notExpired' => $notExpired
        ]);

        // Sync with Accredo system before Magento retrieves the gift card
        // This ensures Magento loads the most up-to-date balance from the database
        try {
            $this->accredoProcessor->syncGiftcard($code, $websiteId);
        } catch (\Exception $e) {
            $this->logger->error('BrittainWynyard_GiftcardAccount: Accredo sync failed in before plugin', [
                'code' => $code,
                'error' => $e->getMessage()
            ]);
            // Don't throw the exception here to avoid breaking the gift card retrieval
        }

        // Return the original arguments (required for before plugins)
        return [$code, $websiteId, $balanceGTE, $onlyEnabled, $notExpired];
    }
}
