<?php
/**
 * BrittainWynyard GiftcardAccount Configuration Helper
 *
 * @package BrittainWynyard_GiftcardAccount
 */

namespace BrittainWynyard\GiftcardAccount\Helper;

use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Helper\Context;
use Magento\Framework\Encryption\EncryptorInterface;
use Magento\Store\Model\ScopeInterface;

/**
 * Configuration helper for Accredo API settings
 */
class Config extends AbstractHelper
{
    /**
     * Configuration path constants
     */
    public const XML_PATH_ACCREDO_ENABLED = 'bw_giftcard_accounts/accredo_integration/enabled';
    public const XML_PATH_ACCREDO_ENDPOINT = 'bw_giftcard_accounts/accredo_integration/endpoint';
    public const XML_PATH_ACCREDO_BEARER_TOKEN = 'bw_giftcard_accounts/accredo_integration/bearer_token';
    public const XML_PATH_ACCREDO_TIMEOUT = 'bw_giftcard_accounts/accredo_integration/timeout';

    public const XML_PATH_AUTO_CREATE_CARDS = 'bw_giftcard_accounts/accredo_integration/auto_create_cards';
    public const XML_PATH_AUTO_UPDATE_BALANCES = 'bw_giftcard_accounts/accredo_integration/auto_update_balances';
    public const XML_PATH_SHOW_AT_CHECKOUT = 'bw_giftcard_accounts/accredo_integration/show_at_checkout';

    /**
     * HTTP header constants
     */
    public const HEADER_FLAG_CHECKOUT_GIFTCARD = 'X-FLAG-CHECKOUT-GIFTCARD';

    /**
     * Default values
     */
    public const DEFAULT_TIMEOUT = 30;
    public const DEFAULT_ENDPOINT_PATH = '/api/v1/asics/gift-card';

    /**
     * @var EncryptorInterface
     */
    private $encryptor;

    /**
     * @param Context $context
     * @param EncryptorInterface $encryptor
     */
    public function __construct(
        Context $context,
        EncryptorInterface $encryptor
    ) {
        parent::__construct($context);
        $this->encryptor = $encryptor;
    }

    /**
     * Check if Accredo integration is enabled
     *
     * @param int|null $storeId
     * @return bool
     */
    public function isAccredoEnabled(?int $storeId = null): bool
    {
        return $this->scopeConfig->isSetFlag(
            self::XML_PATH_ACCREDO_ENABLED,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * Get Accredo API endpoint URL
     *
     * @param int|null $storeId
     * @return string
     */
    public function getAccredoEndpoint(?int $storeId = null): string
    {
        $baseUrl = $this->scopeConfig->getValue(
            self::XML_PATH_ACCREDO_ENDPOINT,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );

        // Ensure URL ends with slash and add the path
        $baseUrl = rtrim($baseUrl ?: '', '/');
        return $baseUrl . self::DEFAULT_ENDPOINT_PATH;
    }

    /**
     * Get Accredo API bearer token (decrypted)
     *
     * @param int|null $storeId
     * @return string
     */
    public function getAccredoBearerToken(?int $storeId = null): string
    {
        $encryptedToken = $this->scopeConfig->getValue(
            self::XML_PATH_ACCREDO_BEARER_TOKEN,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );

        if (!$encryptedToken) {
            return '';
        }

        return $this->encryptor->decrypt($encryptedToken);
    }

    /**
     * Get API request timeout in seconds
     *
     * @param int|null $storeId
     * @return int
     */
    public function getRequestTimeout(?int $storeId = null): int
    {
        $timeout = $this->scopeConfig->getValue(
            self::XML_PATH_ACCREDO_TIMEOUT,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );

        return (int)($timeout ?: self::DEFAULT_TIMEOUT);
    }



    /**
     * Check if cards should be auto-created when found in Accredo
     *
     * @param int|null $storeId
     * @return bool
     */
    public function shouldAutoCreateCards(?int $storeId = null): bool
    {
        return $this->scopeConfig->isSetFlag(
            self::XML_PATH_AUTO_CREATE_CARDS,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * Check if balances should be auto-updated when they differ
     *
     * @param int|null $storeId
     * @return bool
     */
    public function shouldAutoUpdateBalances(?int $storeId = null): bool
    {
        return $this->scopeConfig->isSetFlag(
            self::XML_PATH_AUTO_UPDATE_BALANCES,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * Check if gift card input should be shown at checkout
     *
     * @param int|null $storeId
     * @return bool
     */
    public function shouldShowAtCheckout(?int $storeId = null): bool
    {
        return $this->scopeConfig->isSetFlag(
            self::XML_PATH_SHOW_AT_CHECKOUT,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * Check if the HTTP header flag is set to force show gift card input
     *
     * @return bool
     */
    public function isCheckoutGiftcardHeaderFlagSet(): bool
    {
        $headerKey = 'HTTP_' . str_replace('-', '_', strtoupper(self::HEADER_FLAG_CHECKOUT_GIFTCARD));
        $headerValue = $_SERVER[$headerKey] ?? null;
        return $headerValue === '1';
    }

    /**
     * Determine if gift card should be visible based on config and header override
     *
     * @param int|null $storeId
     * @return bool
     */
    public function shouldGiftcardBeVisible(?int $storeId = null): bool
    {
        // If header flag is set, always show
        if ($this->isCheckoutGiftcardHeaderFlagSet()) {
            return true;
        }

        // Otherwise, use configuration setting
        return $this->shouldShowAtCheckout($storeId);
    }

    /**
     * Get complete configuration as array for debugging
     *
     * @param int|null $storeId
     * @return array
     */
    public function getDebugConfig(?int $storeId = null): array
    {
        return [
            'enabled' => $this->isAccredoEnabled($storeId),
            'endpoint' => $this->getAccredoEndpoint($storeId),
            'has_token' => !empty($this->getAccredoBearerToken($storeId)),
            'timeout' => $this->getRequestTimeout($storeId),

            'auto_create_cards' => $this->shouldAutoCreateCards($storeId),
            'auto_update_balances' => $this->shouldAutoUpdateBalances($storeId),
            'show_at_checkout' => $this->shouldShowAtCheckout($storeId),
            'checkout_header_flag' => $this->isCheckoutGiftcardHeaderFlagSet(),
            'giftcard_visible' => $this->shouldGiftcardBeVisible($storeId)
        ];
    }
}
