<?php

declare(strict_types=1);

namespace BrittainWynyard\CatalogSuperStyle\Plugin\ProductRender;

use BrittainWynyard\CatalogSuperStyle\Helper\Data;
use Magento\Catalog\Api\Data\ProductInterface;
use Magento\Catalog\Api\Data\ProductRenderInterface;
use Magento\Catalog\Ui\DataProvider\Product\ProductRenderCollectorComposite;
use Magento\ConfigurableProduct\Model\Product\Type\Configurable;
use Psr\Log\LoggerInterface;

class BatchSuperSwatchLoader
{
    /**
     * @var Data
     */
    private $dataHelper;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var array Cache to track processed products to avoid duplicate batch loads
     */
    private $processedProducts = [];

    public function __construct(
        Data $dataHelper,
        LoggerInterface $logger
    ) {
        $this->dataHelper = $dataHelper;
        $this->logger = $logger;
    }

    /**
     * Plugin to batch-load super swatch counts before product render collectors run
     *
     * @param ProductRenderCollectorComposite $subject
     * @param ProductInterface $product
     * @param ProductRenderInterface $productRender
     * @return array
     */
    public function beforeCollect(
        ProductRenderCollectorComposite $subject,
        ProductInterface $product,
        ProductRenderInterface $productRender
    ): array {
        // Only process configurable products with super_style_id
        /** @var \Magento\Catalog\Model\Product $product */
        if ($product->getTypeId() === Configurable::TYPE_CODE && $product->getData('super_style_id')) {
            $productId = $product->getId();
            
            // Check if we already processed this product
            if (!isset($this->processedProducts[$productId])) {
                // Load data for this single product (the helper will handle deduplication)
                $this->dataHelper->loadSwatchCounts([$product]);
                
                // Get the count and set it on the product
                /** @var \Magento\Catalog\Model\Product $product */
                $superStyleId = $product->getData('super_style_id');
                if ($superStyleId) {
                    $count = $this->dataHelper->getSuperSwatchCount($superStyleId);
                    $product->setData('super_swatch_related_count', $count);
                    $product->setData('super_swatch_count_loaded', true);
                }
                
                $this->processedProducts[$productId] = true;
            }
        }

        return [$product, $productRender];
    }
}
