<?php

declare(strict_types=1);

namespace BrittainWynyard\CatalogSuperStyle\Model\ResourceModel\Index;

use Magento\Framework\Model\ResourceModel\Db\Collection\AbstractCollection;
use BrittainWynyard\CatalogSuperStyle\Model\Index as Model;
use BrittainWynyard\CatalogSuperStyle\Model\ResourceModel\Index as ResourceModel;

/**
 * Index Collection
 */
class Collection extends AbstractCollection
{
    /**
     * Initialize collection
     *
     * @return void
     */
    protected function _construct()
    {
        $this->_init(Model::class, ResourceModel::class);
    }

    /**
     * Get product counts by super style IDs for a specific website
     *
     * This replaces the raw SQL COUNT query with proper Magento ORM
     *
     * @param array $superStyleIds
     * @param int $websiteId
     * @return array Array with super_style_id as key and count as value
     */
    public function getCountsByStyleIds(array $superStyleIds, int $websiteId): array
    {
        if (empty($superStyleIds)) {
            return [];
        }

        // Reset any previous filters/selections
        $this->clear();

        // Add filters for the specific super style IDs and website
        $this->addFieldToFilter('super_style_id', ['in' => $superStyleIds]);
        $this->addFieldToFilter('website_id', $websiteId);

        // Group by super_style_id and count products
        $this->getSelect()
            ->reset(\Magento\Framework\DB\Select::COLUMNS)
            ->columns([
                'super_style_id',
                'product_count' => new \Zend_Db_Expr('COUNT(*)')
            ])
            ->group('super_style_id');

        // Execute query and return as associative array
        $connection = $this->getConnection();
        return $connection->fetchPairs($this->getSelect());
    }

    /**
     * Get all product IDs grouped by super style IDs
     *
     * Useful for debugging or detailed analysis
     *
     * @param array $superStyleIds
     * @param int $websiteId
     * @return array Array with super_style_id as key and comma-separated product IDs as value
     */
    public function getProductIdsByStyleIds(array $superStyleIds, int $websiteId): array
    {
        if (empty($superStyleIds)) {
            return [];
        }

        // Reset any previous filters/selections
        $this->clear();

        // Add filters
        $this->addFieldToFilter('super_style_id', ['in' => $superStyleIds]);
        $this->addFieldToFilter('website_id', $websiteId);

        // Group by super_style_id and concatenate product IDs
        $this->getSelect()
            ->reset(\Magento\Framework\DB\Select::COLUMNS)
            ->columns([
                'super_style_id',
                'product_ids' => new \Zend_Db_Expr('GROUP_CONCAT(product_id)')
            ])
            ->group('super_style_id');

        // Execute query and return as associative array
        $connection = $this->getConnection();
        return $connection->fetchPairs($this->getSelect());
    }

    /**
     * Get products by super style ID for a specific website
     *
     * @param string $superStyleId
     * @param int $websiteId
     * @return array Array of product IDs
     */
    public function getProductIdsByStyleId(string $superStyleId, int $websiteId): array
    {
        // Reset any previous filters/selections
        $this->clear();

        // Add filters
        $this->addFieldToFilter('super_style_id', $superStyleId);
        $this->addFieldToFilter('website_id', $websiteId);

        // Select only product_id column
        $this->getSelect()
            ->reset(\Magento\Framework\DB\Select::COLUMNS)
            ->columns(['product_id']);

        // Execute query and return array of product IDs
        $connection = $this->getConnection();
        return $connection->fetchCol($this->getSelect());
    }

    /**
     * Get all product IDs for multiple super style IDs
     *
     * This method returns a flat array of all product IDs for the given style IDs
     *
     * @param array $superStyleIds
     * @param int $websiteId
     * @return array Array of product IDs
     */
    public function getAllProductIdsByStyleIds(array $superStyleIds, int $websiteId): array
    {
        if (empty($superStyleIds)) {
            return [];
        }

        // Reset any previous filters/selections
        $this->clear();

        // Add filters
        $this->addFieldToFilter('super_style_id', ['in' => $superStyleIds]);
        $this->addFieldToFilter('website_id', $websiteId);

        // Select only product_id column
        $this->getSelect()
            ->reset(\Magento\Framework\DB\Select::COLUMNS)
            ->columns(['product_id']);


        // Execute query and return array of product IDs
        $connection = $this->getConnection();
        $result = $connection->fetchCol($this->getSelect());
   
        
        return $result;
    }

    /**
     * Clear the collection state for reuse
     *
     * This ensures we can call multiple methods on the same collection instance
     *
     * @return $this
     */
    public function clear(): self
    {
        $this->_reset();
        return $this;
    }
}
