<?php

declare(strict_types=1);

namespace BrittainWynyard\CatalogSuperStyle\Model\Indexer;

use Magento\Framework\Indexer\ActionInterface;
use Magento\Framework\Mview\ActionInterface as MviewActionInterface;
use Magento\Framework\App\ResourceConnection;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Eav\Model\Config as EavConfig;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory as ProductCollectionFactory;
use Magento\Catalog\Model\Product\Attribute\Source\Status;
use Magento\Catalog\Model\Product\Visibility;
use Magento\Framework\DB\Adapter\AdapterInterface;
use Psr\Log\LoggerInterface;

/**
 * Super Style Index - handles all indexing operations
 */
class SuperStyleIndex implements ActionInterface, MviewActionInterface
{
    /**
     * @var ResourceConnection
     */
    private $resourceConnection;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var EavConfig
     */
    private $eavConfig;

    /**
     * @var ProductCollectionFactory
     */
    private $productCollectionFactory;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var Visibility
     */
    private $visibility;

    /**
     * @var array
     */
    private $attributeIds = [];

    /**
     * Constructor
     *
     * @param ResourceConnection $resourceConnection
     * @param StoreManagerInterface $storeManager
     * @param EavConfig $eavConfig
     * @param ProductCollectionFactory $productCollectionFactory
     * @param LoggerInterface $logger
     * @param Visibility $visibility
     */
    public function __construct(
        ResourceConnection $resourceConnection,
        StoreManagerInterface $storeManager,
        EavConfig $eavConfig,
        ProductCollectionFactory $productCollectionFactory,
        LoggerInterface $logger,
        Visibility $visibility
    ) {
        $this->resourceConnection = $resourceConnection;
        $this->storeManager = $storeManager;
        $this->eavConfig = $eavConfig;
        $this->productCollectionFactory = $productCollectionFactory;
        $this->logger = $logger;
        $this->visibility = $visibility;
    }

    /**
     * Execute full indexation
     */
    public function executeFull()
    {
        $this->reindexAll();
    }

    /**
     * Execute partial indexation by ID list
     *
     * @param array $ids
     */
    public function executeList(array $ids)
    {
        $this->reindexList($ids);
    }

    /**
     * Execute partial indexation by ID
     *
     * @param int $id
     */
    public function executeRow($id)
    {
        $this->reindexList([$id]);
    }

    /**
     * Execute by mview changelog
     *
     * @param array|int $ids
     */
    public function execute($ids)
    {
        // Ensure we have an array of integers
        $productIds = is_array($ids) ? array_map(function ($id) {
            return (int) $id;
        }, $ids) : [(int) $ids];
        $this->reindexList($productIds);
    }

    /**
     * Rebuild the entire index
     */
    private function reindexAll(): void
    {
        $connection = $this->resourceConnection->getConnection();
        $indexTable = $this->resourceConnection->getTableName('catalog_product_super_style_index');

        try {
            // Clear existing data
            $connection->truncateTable($indexTable);

            // Get all websites
            $websites = $this->storeManager->getWebsites();

            foreach ($websites as $website) {
                $this->reindexWebsite((int)$website->getId());
            }

        } catch (\Exception $e) {
            $this->logger->error('SuperStyleIndex: Full reindex failed', [
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Reindex specific products
     *
     * @param array $productIds
     */
    private function reindexList(array $productIds): void
    {
        if (empty($productIds)) {
            return;
        }

        try {
            $connection = $this->resourceConnection->getConnection();
            $indexTable = $this->resourceConnection->getTableName('catalog_product_super_style_index');

            // Remove existing entries for these products
            $connection->delete($indexTable, [
                'product_id IN (?)' => $productIds
            ]);

            // Get all websites and reindex for each
            $websites = $this->storeManager->getWebsites();

            foreach ($websites as $website) {
                $this->reindexProductsForWebsite($productIds, (int)$website->getId());
            }

        } catch (\Exception $e) {
            $this->logger->error('SuperStyleIndex: Partial reindex failed', [
                'error' => $e->getMessage(),
                'product_ids' => $productIds
            ]);
            throw $e;
        }
    }

    /**
     * Reindex all products for a specific website
     */
    private function reindexWebsite(int $websiteId): void
    {
        // Pass null for productIds to indicate full reindexing
        $this->reindexProductsForWebsite(null, $websiteId);
    }

    /**
     * Insert batch of data efficiently
     *
     * @param AdapterInterface $connection
     * @param string $table
     * @param array $data
     * @return void
     */
    private function insertBatch(AdapterInterface $connection, string $table, array $data): void
    {
        if (empty($data)) {
            return;
        }

        try {
            // Use INSERT IGNORE to handle duplicates gracefully
            $connection->insertOnDuplicate($table, $data, ['super_style_id', 'product_id', 'website_id']);
            
        } catch (\Exception $e) {
            $this->logger->error('SuperStyleIndex: Batch insert failed', [
                'error' => $e->getMessage(),
                'batch_size' => count($data)
            ]);
            throw $e;
        }
    }

    /**
     * Reindex specific products for a website using ORM
     *
     * @param array|null $productIds
     * @param int $websiteId
     */
    private function reindexProductsForWebsite(?array $productIds, int $websiteId): void
    {
        // For full reindex, productIds will be null
        // For incremental reindex, productIds will contain specific IDs to process
        if ($productIds !== null && empty($productIds)) {
            return;
        }

        $isIncremental = $productIds !== null;
        
        // Log milestone - start of reindex
        if ($isIncremental) {
            $this->logger->info('SuperStyleIndex: Starting incremental reindex', [
                'website_id' => $websiteId,
                'product_count' => count($productIds)
            ]);
        } else {
            $this->logger->info('SuperStyleIndex: Starting full reindex', [
                'website_id' => $websiteId
            ]);
        }

        try {
            // Create product collection
            $collection = $this->productCollectionFactory->create();
            
            // Filter to specific product IDs only for incremental reindex
            if ($isIncremental) {
                $collection->addFieldToFilter('entity_id', ['in' => $productIds]);
            }
            
            // Add website filter
            $collection->addWebsiteFilter($websiteId);
            
            // Add attribute filters using ORM methods
            $collection->addAttributeToFilter('status', Status::STATUS_ENABLED);
            $collection->addAttributeToFilter('visibility', $this->visibility->getVisibleInSiteIds());
            
            // Filter only products with super_style_id
            $collection->addAttributeToFilter('super_style_id', ['neq' => '']);
            $collection->addAttributeToFilter('super_style_id', ['notnull' => true]);
            
            // Select only needed attributes for performance
            $collection->addAttributeToSelect(['super_style_id']);
            
            $connection = $this->resourceConnection->getConnection();
            $indexTable = $this->resourceConnection->getTableName('catalog_product_super_style_index');
            
            $batchData = [];
            $totalProcessed = 0;
            
            // Get collection size before processing
            $collectionSize = $collection->getSize();
            
            if ($isIncremental) {
                
                // For incremental reindex, we need to handle ALL requested products
                // even if they no longer have super_style_id (to remove them from index)
                
                // Create a collection without super_style_id filters to get ALL requested products
                $allProductsCollection = $this->productCollectionFactory->create();
                $allProductsCollection->addFieldToFilter('entity_id', ['in' => $productIds]);
                $allProductsCollection->addWebsiteFilter($websiteId);
                $allProductsCollection->addAttributeToFilter('status', Status::STATUS_ENABLED);
                $allProductsCollection->addAttributeToFilter('visibility', $this->visibility->getVisibleInSiteIds());
                $allProductsCollection->addAttributeToSelect(['super_style_id']);
                
                
                // Process each product from the unfiltered collection
                foreach ($allProductsCollection as $product) {
                    $superStyleId = $product->getData('super_style_id');
                    
                    
                    if ($superStyleId && !empty(trim($superStyleId))) {
                        $batchData[] = [
                            'super_style_id' => $superStyleId,
                            'product_id' => (int)$product->getId(),
                            'website_id' => $websiteId
                        ];
                        $totalProcessed++;
                    }
                    // Note: Products without super_style_id will be removed from index
                    // by the earlier DELETE query in reindexList()
                }
                
                // Insert all data in one batch
                if (!empty($batchData)) {
                    $this->insertBatch($connection, $indexTable, $batchData);
                }
                
            } else {
                // For full reindex, use pagination to avoid memory issues
                
                $pageSize = 1000;
                $currentPage = 1;
                $totalPages = ceil($collectionSize / $pageSize);
                
                while ($currentPage <= $totalPages) {
                    // Create a fresh collection for this page
                    $pageCollection = $this->productCollectionFactory->create();
                    
                    // Apply same filters
                    $pageCollection->addWebsiteFilter($websiteId);
                    $pageCollection->addAttributeToFilter('status', Status::STATUS_ENABLED);
                    $pageCollection->addAttributeToFilter('visibility', $this->visibility->getVisibleInSiteIds());
                    $pageCollection->addAttributeToFilter('super_style_id', ['neq' => '']);
                    $pageCollection->addAttributeToFilter('super_style_id', ['notnull' => true]);
                    $pageCollection->addAttributeToSelect(['super_style_id']);
                    
                    // Set pagination
                    $pageCollection->setPageSize($pageSize);
                    $pageCollection->setCurPage($currentPage);
                    
                    
                    // Process products in this page
                    foreach ($pageCollection as $product) {
                        $superStyleId = $product->getData('super_style_id');
                        if ($superStyleId) {
                            $batchData[] = [
                                'super_style_id' => $superStyleId,
                                'product_id' => (int)$product->getId(),
                                'website_id' => $websiteId
                            ];
                            
                            // Insert in batches
                            if (count($batchData) >= 500) {
                                $this->insertBatch($connection, $indexTable, $batchData);
                                $totalProcessed += count($batchData);
                                $batchData = [];
                            }
                        }
                    }
                    
                    // Clear memory
                    $pageCollection->clear();
                    unset($pageCollection);
                    
                    $currentPage++;
                }
                
                // Insert remaining data
                if (!empty($batchData)) {
                    $this->insertBatch($connection, $indexTable, $batchData);
                    $totalProcessed += count($batchData);
                }
            }
            
            // Log milestone - completion of reindex
            if ($isIncremental) {
                $foundProducts = count($batchData);
                $missingProducts = count($productIds) - $foundProducts;
                
                if ($missingProducts > 0) {
                    $this->logger->warning('SuperStyleIndex: Some products not found or filtered out', [
                        'website_id' => $websiteId,
                        'requested_count' => count($productIds),
                        'found_count' => $foundProducts,
                        'missing_count' => $missingProducts
                    ]);
                }
                
                $this->logger->info('SuperStyleIndex: Incremental reindex completed', [
                    'website_id' => $websiteId,
                    'products_processed' => $foundProducts
                ]);
            } else {
                $this->logger->info('SuperStyleIndex: Full reindex completed', [
                    'website_id' => $websiteId,
                    'total_processed' => $totalProcessed
                ]);
            }
            
        } catch (\Exception $e) {
            $this->logger->error('SuperStyleIndex: Reindex failed', [
                'website_id' => $websiteId,
                'reindex_type' => $isIncremental ? 'incremental' : 'full',
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Get attribute ID by code (cached)
     *
     * @param string $attributeCode
     * @return int
     */
    private function getAttributeId(string $attributeCode): int
    {
        if (!isset($this->attributeIds[$attributeCode])) {
            $attribute = $this->eavConfig->getAttribute('catalog_product', $attributeCode);
            $this->attributeIds[$attributeCode] = (int)$attribute->getAttributeId();
        }

        return $this->attributeIds[$attributeCode];
    }
}
