<?php

namespace Asics\Verifone\Controller\Standard;

use Magento\Framework\App\ResponseInterface;
use Verifone\Hosted\Helper\Verifone;

class WebhookRewrite extends \Verifone\Hosted\Controller\Standard\Webhook
{
    public function execute()
    {
        $input = $this->request->getContent();
        if (!$input) {
            $this->_logger->error('No content received in Verifone webhook');
            return $this->api_response_error('Unable to process data');
        }

        $data = json_decode($input, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            $this->_logger->error(
                __CLASS__ . '@' . __FUNCTION__ . ' line ' . __LINE__ . ': Unable to decode response from API',
                ['response' => $input]
            );
            return $this->api_response_error('Unable to process data');
        }

        $paymentMethod = $this->getPaymentMethod();

        if ($this->stringStartsWith($data['eventType'], 'Checkout')) {
            try {
                $checkout = $this->getCheckoutHelper()->getCheckout($paymentMethod, $data['recordId']);
                $data['recordId'] = $checkout['transaction_id'];
            } catch (\Exception $exception) {
                $this->_logger->error(
                    __CLASS__ . '@' . __FUNCTION__ . ' line ' . __LINE__ . ': There was a request for an order: ',
                    ['data' => $data, 'exception' => $exception]
                );
                return $this->api_response_error('Unable to process data');
            }
        }

        $response = $this->getCheckoutHelper()->getTransaction($paymentMethod, $data['recordId']);
        $transactionId = $response['id'];
        $amountPaid = $this->_checkoutHelper->convertAmountToVerifone($response['amount'], $response['currency_code']);

        if (!empty($response['merchant_reference'])) {
            $quote = $this->_quoteRepository->get($response['merchant_reference']);
        } elseif (!empty($response['id'])) {
            $quote = $this->_quoteRepository->get($response['id']);
        } else {
            $this->_logger->error(
                __CLASS__ . '@' . __FUNCTION__ . ' line ' . __LINE__ . ': API did not sent the ID of the quote',
                ['response' => $response]
            );
            return $this->api_response_error('Unable to process data');
        }

        $order = $this->getCheckoutHelper()->getOrderByIncrementId($quote->getReservedOrderId());

        switch ($response['status']) {
            case Verifone::VERIFONE_TRANSACTION_CAPTURE_AUTHORISED:
            case Verifone::VERIFONE_TRANSACTION_CAPTURE_SETTLED:
            case Verifone::VERIFONE_TRANSACTION_AUTHORISED:
            case Verifone::VERIFONE_TRANSACTION_SALE_AUTHORISED:
            case Verifone::VERIFONE_TRANSACTION_SALE_SETTLED:
            case Verifone::VERIFONE_TRANSACTION_AUTHORISATION_AUTHORISED:
            case Verifone::VERIFONE_TRANSACTION_CARD_VERIFICATION_AUTHORISED:
                if (!$order) {
                    $code = $paymentMethod->getCode();
                    $quote->setPaymentMethod($code);
                    $quote->getPayment()->importData(['method' => $code]);
                    $this->_quoteRepository->save($quote);

                    try {
                        $this->_cartManagement->placeOrder(
                            $quote->getId(),
                            $quote->getPayment()
                        );
                        $order = $this->getOrder();
                    } catch (\Exception $e) {
                        $this->_logger->error($e->getMessage());
                        $this->_logger->error($e->getTraceAsString());
                        if (isset($this->messageManager)) {
                            $this->messageManager->addExceptionMessage($e, __('We can\'t place the order.'));
                        }
                        $this->_logger->error(
                            'Unable to place Verifone order via webhook',
                            ['quote_id' => $quote->getId(), 'exception' => $e]
                        );
                        return $this->api_response_error('Unable to place order');
                    }
                }
                $this->getCheckoutSession()
                    ->setLastSuccessQuoteId($quote->getId())
                    ->setLastQuoteId($quote->getId())
                    ->setLastRealOrderId($order->getIncrementId())
                    ->setLastOrderId($order->getId())
                    ->setLastOrderStatus($order->getStatus());

                /** Because with a refund, we can receive a transaction event but also a checkout event having transaction_id the original successful transaction, check that we don't already have a refunded amount before set status to complete.
                 */
                if (!$this->_checkoutHelper->checkVerifoneTransactionIdInOrderLogHistory($transactionId, $order) && !($order->getTotalRefunded() ?? 0)) {
                    $payment = $order->getPayment();
                    $paymentMethod = $payment->getMethodInstance();
                    $this->_checkoutHelper->setUpVerifoneSettings($paymentMethod);

                    $order->setExtOrderId($response['id']);
                    $order->save();

                    $manualCapture = $this->getCheckoutHelper()->checkManualCapture();
                    $response['transaction_id'] = $transactionId;
                    $paymentMethod->postProcessing($order, ['transaction_id' => $transactionId], $manualCapture, $response['payment_product_type'], (int) $amountPaid);
                    $order->save();
                }
                break;

            case Verifone::VERIFONE_TRANSACTION_PENDING:
                if (!$order) {
                    break;
                }

                $order->setExtOrderId($response['id'])
                    ->addCommentToStatusHistory(__('The payment is not done yet (transaction is in PRE/FINAL AUTH status), use the CAPTURE button to fully/partial capture the amount.
                    Attempting to refund the order in part or in full, before capturing, will void the authorization and cancel the payment.'))
                    ->save();
                break;

            default:
                if (!$order) {
                    break;
                }

                $order->addCommentToStatusHistory(__('Transaction event logged from Verifone: ') . $data['eventType'] . ' Response status: ' . $response['status'])
                    ->save();
        }

        return $this->api_response_ok();
    }

    public function api_response_error($message = null): ResponseInterface
    {
        $this->api_response($message, 500);
        return $this->response;
    }

    public function api_response($message = null, $statusCode = 200)
    {
        $this->response->setStatusCode($statusCode);
        $this->response->setContent($message);
        return;
    }
}
