<?php

// Magento\Framework\DataObject implements the magic call function

// Create payment module http://www.josephmcdermott.co.uk/basics-creating-magento2-payment-method
// https://github.com/magento/magento2-samples/tree/master/sample-module-payment-provider
namespace Windcave\Payments\Model;

use \Magento\Framework\Exception\State\InvalidTransitionException;

class Payment extends \Magento\Payment\Model\Method\AbstractMethod
{
    /**
     *
     * @var \Magento\Framework\App\ObjectManager
     */
    private $_objectManager;

    /**
     * @var string
     */
    protected $_code;
    
    /**
     * @var string
     */
    protected $_infoBlockType = \Windcave\Payments\Block\Info::class;

    /**
     * @var bool
     */
    protected $_isGateway = true;

    /**
     * @var bool
     */
    protected $_canAuthorize = true;

    /**
     * @var bool
     */
    protected $_canCapture = true;

    /**
     * @var bool
     */
    protected $_canUseInternal = false;

    /**
     * @var bool
     */
    protected $_canUseCheckout = true;

    /**
     * @var bool
     */
    protected $_canUseForMultishipping = false;

    /**
     * @var bool
     */
    protected $_canRefund = true;

    /**
     * @var bool
     */
    protected $_canRefundInvoicePartial = true;

    /**
     * @var bool
     */
    protected $_isInitializeNeeded = true;

    /**
     * @var \Windcave\Payments\Model\PaymentHelper
     */
    private $_paymentHelper;

    /**
     * @var \Windcave\Payments\Model\Api\ApiPxPayHelper
     */
    private $_apiPxPayHelper;

    /**
     * @var \Magento\Framework\Notification\NotifierInterface
     */
    private $_notifierInterface;

    public const PXPAY_CODE = "windcave_pxpay2";

    /**
     * Constructor
     *
     * @param \Magento\Framework\Model\Context $context
     * @param \Magento\Framework\Registry $registry
     * @param \Magento\Framework\Api\ExtensionAttributesFactory $extensionFactory
     * @param \Magento\Framework\Api\AttributeValueFactory $customAttributeFactory
     * @param \Magento\Payment\Helper\Data $paymentData
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     * @param \Magento\Payment\Model\Method\Logger $logger
     * @param \Magento\Framework\Notification\NotifierInterface $notifierInterface
     * @param \Magento\Framework\Model\ResourceModel\AbstractResource $resource
     * @param \Magento\Framework\Data\Collection\AbstractDb $resourceCollection
     * @param array $data
     */
    public function __construct(
        \Magento\Framework\Model\Context $context,
        \Magento\Framework\Registry $registry,
        \Magento\Framework\Api\ExtensionAttributesFactory $extensionFactory,
        \Magento\Framework\Api\AttributeValueFactory $customAttributeFactory,
        \Magento\Payment\Helper\Data $paymentData,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Payment\Model\Method\Logger $logger,
        \Magento\Framework\Notification\NotifierInterface $notifierInterface,
        \Magento\Framework\Model\ResourceModel\AbstractResource $resource = null,
        \Magento\Framework\Data\Collection\AbstractDb $resourceCollection = null,
        array $data = []
    ) {
        parent::__construct(
            $context,
            $registry,
            $extensionFactory,
            $customAttributeFactory,
            $paymentData,
            $scopeConfig,
            $logger,
            $resource,
            $resourceCollection,
            $data
        );
        $this->_code = self::PXPAY_CODE;
        $this->_objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $this->_logger = $this->_objectManager->get(\Windcave\Payments\Logger\DpsLogger::class);
        $this->_logger->info(__METHOD__);
        $this->_apiPxPayHelper = $this->_objectManager->get(\Windcave\Payments\Model\Api\ApiPxPayHelper::class);
        $configuration = $this->_objectManager->get(\Windcave\Payments\Helper\Configuration::class);
        $communication = $this->_objectManager->get(\Windcave\Payments\Helper\Communication::class);
        $this->_paymentHelper = $this->_objectManager->create(\Windcave\Payments\Model\PaymentHelper::class);
        $this->_paymentHelper->init($configuration, $communication);
        $this->_notifierInterface = $notifierInterface;
        $this->_canCapturePartial = $this->getConfigData('can_capture_partial');
    }

    /**
     * @inheritdoc
     */
    public function assignData(\Magento\Framework\DataObject $data)
    {
        $this->_logger->info(__METHOD__ . " data:" . var_export($data, true));
        $infoInstance = $this->getInfoInstance();

        $info = $infoInstance->getAdditionalInformation();
        if (isset($info) && !empty($info['DpsTxnRef'])) {
            $this->_logger->info(__METHOD__ . " payment finished. DpsTxnRef:" . $info['DpsTxnRef']);
            return $this; //The transaction is processed.
        }
        
        $source = $data;
        if (isset($data['additional_data'])) {
            $source = $this->_objectManager->create(\Magento\Framework\DataObject::class);
            $source->setData($data['additional_data']);
        }
        
        $info = [];
        $dpsBillingId = $source->getData("windcave_billingId");
        $info["UseSavedCard"] = filter_var($source->getData("windcave_useSavedCard"), FILTER_VALIDATE_BOOLEAN);
        $info["EnableAddBillCard"] = filter_var(
            $source->getData("windcave_enableAddBillCard"),
            FILTER_VALIDATE_BOOLEAN
        );
        if (isset($dpsBillingId) && !empty($dpsBillingId)) {
            $info["DpsBillingId"] = $dpsBillingId;
            $info["UseSavedCard"] = true;
            
            $info["EnableAddBillCard"] = false; // Do not add billing token when rebill.
        } else {
            $info["UseSavedCard"] = false;
        }
        $info["cartId"] = $source->getData("cartId");
        $info["guestEmail"] = $source->getData("guestEmail");
        $info["DpsHandler"] = "8";
        
        $infoInstance->setAdditionalInformation($info);
        $infoInstance->save();

        return $this;
    }
    
    /**
     * @inheritdoc
     */
    public function getConfigPaymentAction()
    {
        return $this->_paymentHelper->getConfigPaymentAction($this->getStore());
    }
    
    /**
     * @inheritdoc
     */
    public function canCapture()
    {
        return $this->_canCapture;
    }
    
    /**
     * @inheritdoc
     */
    public function capture(\Magento\Payment\Model\InfoInterface $payment, $amount)
    {
        $this->_logger->info(__METHOD__ . " payment amount:" . $amount);
        $this->_paymentHelper->capture($payment, $amount, $this->getStore());
        return $this;
    }

    /**
     * @inheritdoc
     */
    public function canVoid()
    {
        $payment = $this->getInfoInstance();
        $this->_canVoid =  $this->_paymentHelper->canVoid($this->getStore(), $payment);
        return $this->_canVoid;
    }

    /**
     * @inheritdoc
     */
    public function void(\Magento\Payment\Model\InfoInterface $payment)
    {
        $this->_logger->info(__METHOD__);
        parent::void($payment);
        
        $this->_paymentHelper->void($payment, $this->getStore());
        return $this;
    }

    /**
     * @inheritdoc
     */
    public function cancel(\Magento\Payment\Model\InfoInterface $payment)
    {
        $this->_logger->info(__METHOD__);
        $this->_paymentHelper->void($payment, $this->getStore());
        return $this;
    }

    /**
     * @inheritdoc
     */
    public function canRefund()
    {
        return $this->_canRefund;
    }

    /**
     * @inheritdoc
     */
    public function canRefundPartialPerInvoice()
    {
        return $this->_canRefundInvoicePartial;
    }

    /**
     * @inheritdoc
     */
    public function refund(\Magento\Payment\Model\InfoInterface $payment, $amount)
    {
        // Mage_Sales_Model_Order_Payment::refund
        // use getInfoInstance to get object of Mage_Payment_Model_Info
        // (Mage_Payment_Model_Info::getMethodInstance Mage_Sales_Model_Order_Payment is subclass
        // of Mage_Payment_Model_Info)
        $this->_logger->info(__METHOD__);
        $this->_paymentHelper->refund($payment, $amount, $this->getStore());
        return $this;
    }

    /**
     * @inheritdoc
     */
    public function isAvailable(\Magento\Quote\Api\Data\CartInterface $quote = null)
    {
        $this->_logger->info(__METHOD__);
        return $this->_paymentHelper->isAvailable($quote);
    }

    /**
     * @inheritdoc
     */
    public function isInitializeNeeded()
    {
        $payment = $this->getInfoInstance();
        $dpsHandler = $payment->getAdditionalInformation("DpsHandler");
        if (!isset($dpsHandler) || $dpsHandler != "8") {
            return false;
        }

        return parent::isInitializeNeeded();
    }

    /**
     * Method that will be executed instead of authorize or capture if flag isInitializeNeeded set to true
     *
     * @param string $paymentAction
     * @param object $stateObject
     * @return $this
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function initialize($paymentAction, $stateObject)
    {
        parent::initialize($paymentAction, $stateObject);

        $payment = $this->getInfoInstance();
        $order = $payment->getOrder();

        $additionalInfo = $payment->getAdditionalInformation();
        unset($additionalInfo["DpsHandler"]);

        $this->_logger->info(__METHOD__ . " Attempting to generate PxPay link");
        try {
            $pxPayUrl = $this->_apiPxPayHelper->createUrl($order);
            $additionalInfo["PxPayHPPUrl"] = $pxPayUrl;
        } catch (InvalidTransitionException $exception) {
            // TODO: need to do something here
            $quoteId = $order->getQuoteId();
            $this->_notifierInterface->addMajor(
                "Failed to generate PxPay 2.0 session",
                "OrderId: " . $order->getRealOrderId() . ", QuoteId: " . $quoteId .
                ". See Windcave extension log for more details."
            );

            throw new \Magento\Framework\Exception\LocalizedException(
                __("Internal error while processing quote #{$quoteId}. Please contact support.")
            );
        }
        $stateObject->setState(\Magento\Sales\Model\Order::STATE_PENDING_PAYMENT);
        $stateObject->setStatus($this->getConfigData('order_status'));
        $stateObject->setIsNotified(false);
        $this->_logger->info(__METHOD__ . " Order status changed to pending payment");

        $order->setCanSendNewEmailFlag(false);
        $order->save();

        $this->_logger->info(__METHOD__ . "  realOrderId:" . $order->getRealOrderId() . " Id:" . $order->getId());

        $payment->unsAdditionalInformation();
        $payment->setAdditionalInformation($additionalInfo);

        return $this;
    }
}
