<?php

namespace Windcave\Payments\Model\ApplePay;

use Magento\Payment\Model\Method\AbstractMethod;

//NOTE: AbstractMethod is already deprecated, next implementation should use MethodInterface
// https://magento.stackexchange.com/questions/199027/magento-2-reason-behind-deprecation-of-payment-method-class
class Payment extends AbstractMethod
{
    /**
     * @var \Magento\Framework\App\ObjectManager
     */
    private $_objectManager;

    /**
     * @var \Magento\Framework\Notification\NotifierInterface
     */
    private $_notifierInterface;

    public const APPLEPAY_CODE = 'windcave_applepay';

    /**
     * @var string
     */
    protected $_infoBlockType = \Windcave\Payments\Block\Info::class;

    /**
     * @var bool
     */
    protected $_isGateway = true;

    /**
     * @var bool
     */
    protected $_canAuthorize = true;

    /**
     * @var bool
     */
    protected $_canCapture = true;

    /**
     * @var bool
     */
    protected $_canCapturePartial = true;

    /**
     * @var bool
     */
    protected $_canUseInternal = false;

    /**
     * @var bool
     */
    protected $_canUseCheckout = true;

    /**
     * @var bool
     */
    protected $_canUseForMultishipping = false;

    /**
     * @var bool
     */
    protected $_canRefund = true;

    /**
     * @var bool
     */
    protected $_canRefundInvoicePartial = true;

    /**
     * @var bool
     */
    protected $_isInitializeNeeded = true;

     /**
      * @var \Windcave\Payments\Helper\PaymentUtil
      */
    protected $_paymentUtil;

    /**
     * @var string
     */
    protected $_code = 'windcave_applepay';

    /**
     * @var \Windcave\Payments\Helper\ApplePay\Configuration
     */
    protected $_configuration;
    
    /**
     * @var \Windcave\Payments\Helper\PxFusion\Communication
     */
    protected $_communication;

    /**
     * @var \Magento\Quote\Model\QuoteRepository
     */
    protected $_quoteRepository;

    /**
     * Constructor
     *
     * @param \Magento\Framework\Model\Context $context
     * @param \Magento\Framework\Registry $registry
     * @param \Magento\Framework\Api\ExtensionAttributesFactory $extensionFactory
     * @param \Magento\Framework\Api\AttributeValueFactory $customAttributeFactory
     * @param \Magento\Payment\Helper\Data $paymentData
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     * @param \Magento\Payment\Model\Method\Logger $logger
     * @param \Magento\Quote\Model\QuoteRepository $quoteRepository
     * @param \Magento\Framework\Url $url
     * @param \Magento\Sales\Model\Order\Status\HistoryFactory $orderHistoryFactory
     * @param \Magento\Sales\Model\Order\Payment\Transaction\BuilderInterface $txnBuilder
     * @param \Magento\Framework\Notification\NotifierInterface $notifierInterface
     * @param \Magento\Framework\Model\ResourceModel\AbstractResource $resource
     * @param \Magento\Framework\Data\Collection\AbstractDb $resourceCollection
     * @param array $data
     */
    public function __construct(
        \Magento\Framework\Model\Context $context,
        \Magento\Framework\Registry $registry,
        \Magento\Framework\Api\ExtensionAttributesFactory $extensionFactory,
        \Magento\Framework\Api\AttributeValueFactory $customAttributeFactory,
        \Magento\Payment\Helper\Data $paymentData,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Payment\Model\Method\Logger $logger,
        \Magento\Quote\Model\QuoteRepository $quoteRepository,
        \Magento\Framework\Url $url,
        \Magento\Sales\Model\Order\Status\HistoryFactory $orderHistoryFactory,
        \Magento\Sales\Model\Order\Payment\Transaction\BuilderInterface $txnBuilder,
        \Magento\Framework\Notification\NotifierInterface $notifierInterface,
        \Magento\Framework\Model\ResourceModel\AbstractResource $resource = null,
        \Magento\Framework\Data\Collection\AbstractDb $resourceCollection = null,
        array $data = []
    ) {
        parent::__construct(
            $context,
            $registry,
            $extensionFactory,
            $customAttributeFactory,
            $paymentData,
            $scopeConfig,
            $logger,
            $resource,
            $resourceCollection,
            $data
        );
        $this->_objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $this->_logger = $this->_objectManager->get(\Windcave\Payments\Logger\DpsLogger::class);
        $this->_quoteRepository = $quoteRepository;
        $this->_configuration = $this->_objectManager->get(\Windcave\Payments\Helper\ApplePay\Configuration::class);
        $this->_communication = $this->_objectManager->get(\Windcave\Payments\Helper\ApplePay\Communication::class);
        $this->_paymentUtil = $this->_objectManager->get(\Windcave\Payments\Helper\PaymentUtil::class);

        $this->_notifierInterface = $notifierInterface;

        $this->_logger->info(__METHOD__);
    }
    
    /**
     * Check whether payment method can be used
     *
     * @param \Magento\Quote\Api\Data\CartInterface|null $quote
     * @return bool
     */
    public function isAvailable(\Magento\Quote\Api\Data\CartInterface $quote = null)
    {
        $this->_logger->info(__METHOD__);
        if ($quote != null) {
            $enabled =
                $this->_configuration->getEnabled($quote->getStoreId())
                && $this->_configuration->isValidForApplePay($quote->getStoreId());
        } else {
            $enabled = $this->_configuration->getEnabled() && $this->_configuration->isValidForApplePay();
        }
        $this->_logger->info(__METHOD__ . " enabled:" . $enabled);
        return $enabled;
    }

    /**
     * @inheritdoc
     */
    public function assignData(\Magento\Framework\DataObject $data)
    {
        // invoked by Magento\Quote\Model\PaymentMethodManagement::set
        $this->_logger->info(__METHOD__ . " data:" . var_export($data, true));
        $infoInstance = $this->getInfoInstance();
        $info = $infoInstance->getAdditionalInformation();

        $source = $data;
        if (isset($data['additional_data'])) {
            $source = $this->_objectManager->create(\Magento\Framework\DataObject::class);
            $source->setData($data['additional_data']);
        }

        $info = [
            "paymentData" => $source->getData('paymentData'),
            "transactionId" => $source->getData('transactionId'),
            "cartId" => $source->getData('cartId'),
            "guestEmail" => $source->getData('guestEmail'),
            "paymentType" => "Purchase" //temporary hardcoded
        ];

        $infoInstance->setAdditionalInformation($info);
        $infoInstance->save();
        //$this->_logger->info(__METHOD__ . " saved additional information");
        //$this->_logger->info(__METHOD__ . " data:" . var_export($info, true));
        
        return $this;
    }

    /**
     * @inheritdoc
     */
    public function getConfigPaymentAction()
    {
        // invoked by Magento\Sales\Model\Order\Payment::place
        $this->_logger->info(__METHOD__);
        $paymentType = $this->_configuration->getPaymentType($this->getStore());
        $paymentAction = "";
        
        if ($paymentType == \Windcave\Payments\Model\Config\Source\PaymentOptions::PURCHASE) {
            $paymentAction = \Magento\Payment\Model\Method\AbstractMethod::ACTION_AUTHORIZE_CAPTURE;
        }
        if ($paymentType == \Windcave\Payments\Model\Config\Source\PaymentOptions::AUTH) {
            $paymentAction = \Magento\Payment\Model\Method\AbstractMethod::ACTION_AUTHORIZE;
        }
        $this->_logger->info(__METHOD__ . " paymentType: {$paymentType} paymentAction: {$paymentAction}");
        return $paymentAction;
    }

    /**
     * @inheritdoc
     */
    public function canCapture()
    {
        $this->_logger->info(__METHOD__);

        return $this->_canCapture;
    }

    /**
     * @inheritdoc
     */
    public function capture(\Magento\Payment\Model\InfoInterface $payment, $amount)
    {
        // Invoked by Mage_Sales_Model_Order_Payment::capture
        $this->_logger->info(__METHOD__ . " payment amount:" . $amount);
        $storeId = $this->getStore();
        $orderId = "unknown";
        $order = $payment->getOrder();
        if ($order) {
            $orderId = $order->getIncrementId();
        } else {
            $errorMessage = "Failed to find the order from payment to capture.";
            $this->_logger->critical(__METHOD__ . $errorMessage);
            throw new \Magento\Framework\Exception\PaymentException(
                __("{$errorMessage}. Please refer to Windcave module log for more details.")
            );
        }
        if (!$payment->hasAdditionalInformation()) {
            $this->_logger->info(__METHOD__ . " orderId:{$orderId} additional_information is empty");
        }
        
        $info = $payment->getAdditionalInformation();

        $transactionId = $info["DpsTxnRef"]; // ensure it is unique
        $isPurchase = $info["DpsTransactionType"] == "Purchase";

        if (!$isPurchase) {
            if (!$this->_configuration->isValidForPxPost($storeId)) {
                throw new \Magento\Framework\Exception\PaymentException(
                    __("Windcave module is misconfigured. Please check the configuration before proceeding")
                );
            }

            $currency = $order->getOrderCurrencyCode();
            $dpsTxnRef = $info["DpsTxnRef"];
            $responseXml = $this->_communication->complete($amount, $currency, $dpsTxnRef, $storeId);

            $responseXmlElement = simplexml_load_string($responseXml);
            $this->_logger->info(__METHOD__ . "  responseXml:" . $responseXml);
            if (!$responseXmlElement) {
                $this->_paymentUtil->saveInvalidResponse($payment, $responseXml);
                $errorMessage = "Failed to capture order:{$orderId}, response from Windcave: {$responseXml}";
                $this->_logger->critical(__METHOD__ . $errorMessage);

                throw new \Magento\Framework\Exception\PaymentException(
                    __("Failed to capture the order #{$orderId}. Please refer to Windcave module log for more details.")
                );
            }
            $this->_paymentUtil->savePxPostResponse($payment, $responseXmlElement);
            if (!$responseXmlElement->Transaction || $responseXmlElement->Transaction["success"] != "1") {
                // $this->_paymentUtil->savePxPostResponse($payment, $responseXmlElement);
                $errorMessage = "Failed to capture order:{$orderId}, response from Windcave: {$responseXml}";
                $this->_logger->critical(__METHOD__ . $errorMessage);
                throw new \Magento\Framework\Exception\PaymentException(
                    __("Failed to capture the order #{$orderId}. Please refer to Windcave module log for more details.")
                );
            }
            $transactionId = (string)$responseXmlElement->DpsTxnRef; // use the DpsTxnRef of Complete
        }

        $payment->setTransactionId($transactionId);
        $payment->setIsTransactionClosed(1);
        $payment->setTransactionAdditionalInfo(\Magento\Sales\Model\Order\Payment\Transaction::RAW_DETAILS, $info);
        
        return $this;
    }

    /**
     * @inheritdoc
     */
    public function canRefund()
    {
        $this->_logger->info(__METHOD__);

        return $this->_canRefund;
    }

    /**
     * @inheritdoc
     */
    public function canRefundPartialPerInvoice()
    {
        $this->_logger->info(__METHOD__);

        return $this->_canRefundInvoicePartial;
    }

    /**
     * @inheritdoc
     */
    public function refund(\Magento\Payment\Model\InfoInterface $payment, $amount)
    {
        $this->_logger->info(__METHOD__);

        $storeId = $this->getStore();
        if (!$this->_configuration->isValidForPxPost($storeId)) {
            throw new \Magento\Framework\Exception\PaymentException(
                __("Windcave module is misconfigured. Please check the configuration before proceeding")
            );
        }

        $orderId = "unknown";
        $this->_logger->info(__METHOD__ . " start process.");
        $order = $payment->getOrder();
        if ($order) {
            $orderId = $order->getIncrementId();
            $this->_logger->info(__METHOD__ . " orderId:{$orderId}");
        } else {
            $errorMessage = "Failed to find the order from payment for refund.";
            $this->_logger->critical(__METHOD__ . $errorMessage);
            throw new \Magento\Framework\Exception\PaymentException(
                __("{$errorMessage}. Please refer to Windcave module log for more details.")
            );
        }

        $info = $payment->getAdditionalInformation();
        $isAuth = $info["DpsTransactionType"] == "Auth";
        $this->_logger->info(__METHOD__ . " isAuth = {$isAuth}");
        if ($isAuth) {
            $dpsTxnRef = $payment->getParentTransactionId();
            $this->_logger->info(__METHOD__ . " dpsTxnRef = {$dpsTxnRef}");
        } else {
            $dpsTxnRef = $this->_paymentUtil->findDpsTxnRefForRefund($payment->getAdditionalInformation());
            $this->_logger->info(__METHOD__ . " 2.2 dpsTxnRef = {$dpsTxnRef}");
        }

        $currency = $order->getOrderCurrencyCode();
        if (!$dpsTxnRef) {
            $errorMessage = "Cannot issue a refund for the order #{$orderId}, as the payment has not been captured.";
            $this->_logger->critical(__METHOD__ . $errorMessage);
            throw new \Magento\Framework\Exception\PaymentException(__($errorMessage));
        }
        $this->_logger->info(
            __METHOD__ . " orderId:{$orderId} dpsTxnRef:{$dpsTxnRef} amount:{$amount} currency:{$currency}"
        );
        
        $responseXml = $this->_communication->refund($amount, $currency, $dpsTxnRef, $storeId);
        $responseXmlElement = simplexml_load_string($responseXml);
        // TODO: refund occurs inside the DB transaction. So throwing the exception triggers transaction rollback.
        // That means that _addTransaction doesn't work.
        if (!$responseXmlElement) {
            $errorMessage = " Failed to refund order:{$orderId}, Response from Windcave: {$responseXml}";
            $this->_logger->critical(__METHOD__ . $errorMessage);

            throw new \Magento\Framework\Exception\PaymentException(
                __("Failed to refund the order #{$orderId}. Please refer to Windcave module log for more details.")
            );
        }
        $this->_logger->info(__METHOD__ . " 7.0");

        if (!$responseXmlElement->Transaction || $responseXmlElement->Transaction["success"] != "1") {
            $errorMessage = " Failed to refund order:{$orderId}. Response from Windcave: {$responseXml}";
            $this->_logger->critical(__METHOD__ . $errorMessage);
            
            $message = $this->getErrorMessage(
                "Refund",
                $responseXmlElement,
                ". Please refer to Windcave module log for more details."
            );
            throw new \Magento\Framework\Exception\PaymentException(__($message));
        }
        $payment->setTransactionAdditionalInfo("DpsTxnRef", (string)$responseXmlElement->DpsTxnRef);
        $this->_paymentUtil->savePxPostResponse($payment, $responseXmlElement);
        $this->_logger->info(__METHOD__ . " Done");
        return $this;
    }

    /**
     * @inheritdoc
     */
    public function canVoid()
    {
        $this->_logger->info(__METHOD__);

        $payment = $this->getInfoInstance();
        $order = $payment->getOrder();

        $isAuth = ($payment->getAdditionalInformation("DpsTransactionType") == "Auth");

        $orderState = $order->getState();
        $orderStatus = $order->getStatus();
        if ($isAuth && $orderState == \Magento\Sales\Model\Order::STATE_PENDING_PAYMENT
            //status is no longer reliable for auth as it can also set as processing preferred by merchant.
            //&& $orderStatus == \Windcave\Payments\Controller\PxFusion\CommonAction::STATUS_AUTHORIZED
        ) {
            $this->_canVoid = true;
        } else {
            $this->_canVoid = false;
        }
        return $this->_canVoid;
    }
    
    /**
     * @inheritdoc
     */
    public function void(\Magento\Payment\Model\InfoInterface $payment)
    {
        $this->_logger->info(__METHOD__);

        parent::void($payment);
        $storeId = $this->getStore();
        if (!$this->_configuration->isValidForPxPost($storeId)) {
            throw new \Magento\Framework\Exception\PaymentException(
                __("Windcave module is misconfigured. Please check the configuration before proceeding")
            );
        }

        $orderId = "unknown";
        $order = $payment->getOrder();
        if ($order) {
            $orderId = $order->getIncrementId();
        } else {
            $errorMessage = "Failed to find the order from payment.";
            $this->_logger->critical(__METHOD__ . $errorMessage);
            throw new \Magento\Framework\Exception\PaymentException(
                __("{$errorMessage}. Please refer to Windcave module log for more details.")
            );
        }
        if (!$payment->hasAdditionalInformation()) {
            $this->_logger->info(__METHOD__ . " orderId:{$orderId} additional_information is empty");
        }
        
        $info = $payment->getAdditionalInformation();
        $isAuth = $info["DpsTransactionType"] == "Auth";
        $orderState = $order->getState();
        $orderStatus = $order->getStatus();
        // We will reuse the PxFusion Status
        if ($isAuth && $orderState == \Magento\Sales\Model\Order::STATE_PENDING_PAYMENT
            //status is no longer reliable for auth as it can also set as processing preferred by merchant.
            //&& $orderStatus == \Windcave\Payments\Controller\PxFusion\CommonAction::STATUS_AUTHORIZED
        ) {
            $dpsTxnRef = $info["DpsTxnRef"];
            $responseXml = $this->_communication->void($dpsTxnRef, $storeId);
            $responseXmlElement = simplexml_load_string($responseXml);
            $this->_logger->info(__METHOD__ . "  responseXml:" . $responseXml);
            if (!$responseXmlElement) {
                $this->_paymentUtil->saveInvalidResponse($payment, $responseXml);
                $errorMessage = "Failed to void the order:{$orderId}, response from Windcave: {$responseXml}";
                $this->_logger->critical(__METHOD__ . $errorMessage);
                throw new \Magento\Framework\Exception\PaymentException(
                    __("Failed to void the order #{$orderId}. Please refer to Windcave module log for more details.")
                );
            }
            $this->_paymentUtil->savePxPostResponse($payment, $responseXmlElement);
            if (!$responseXmlElement->Transaction || $responseXmlElement->Transaction["success"] != "1") {
                //$this->_paymentUtil->savePxPostResponse($payment, $responseXmlElement);
                $errorMessage =
                    "Failed to void the order:{$orderId}. ReCo:" . $responseXmlElement->ReCo .
                    "  ResponseText:" . $responseXmlElement->ResponseText . "  AcquirerReCo:" .
                    $responseXmlElement->Transaction->AcquirerReCo . "  AcquirerResponseText:" .
                    $responseXmlElement->Transaction->AcquirerResponseText;
                $this->_logger->critical(__METHOD__ . $errorMessage);
                throw new \Magento\Framework\Exception\PaymentException(__($errorMessage));
            }
        }
        return $this;
    }

    /**
     * @inheritdoc
     */
    public function cancel(\Magento\Payment\Model\InfoInterface $payment)
    {
        $this->_logger->info(__METHOD__);
        $this->void($payment);
        return $this;
    }

    /**
     * Method that will be executed instead of authorize or capture if flag isInitializeNeeded set to true
     *
     * @param string $paymentAction
     * @param object $stateObject
     * @return $this
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function initialize($paymentAction, $stateObject)
    {
        parent::initialize($paymentAction, $stateObject);
        $payment = $this->getInfoInstance();
        $order = $payment->getOrder();
        $stateObject->setState(\Magento\Sales\Model\Order::STATE_PENDING_PAYMENT);
        $stateObject->setStatus($this->getConfigData('order_status'));
        $stateObject->setIsNotified(false);
        $this->_logger->info(__METHOD__ . " Order status changed to pending");
        $order->setCanSendNewEmailFlag(false);
        $order->save();
        return $this;
    }

    /**
     * Builds a callback redirect URL
     *
     * @return string
     */
    private function _buildReturnUrl()
    {
        $this->_logger->info(__METHOD__);
        $url = $this->_url->getUrl('pxpay2/applepay/result', ['_secure' => true]);
        $this->_logger->info(__METHOD__ . " url: {$url} ");
        return $url;
    }
}
