<?php
namespace Windcave\Payments\Model\Api;

use \Magento\Framework\Exception\State\InvalidTransitionException;

class ApiPxPayIFrameHelper
{
    /**
     * @var \Windcave\Payments\Model\Api\ApiCommonHelper
     */
    private $_apiCommonHelper;
    
    /**
     * @var \Windcave\Payments\Helper\PxPayIFrame\UrlCreator
     */
    private $_pxpayUrlCreator;

    /**
     * @var \Windcave\Payments\Logger\DpsLogger
     */
    private $_logger;

    /**
     * @var \Magento\Quote\Model\QuoteFactory
     */
    private $_quoteFactory;

    /**
     * Constructor
     *
     * @param \Magento\Quote\Model\QuoteFactory $quoteFactory
     */
    public function __construct(\Magento\Quote\Model\QuoteFactory $quoteFactory)
    {
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $this->_pxpayUrlCreator = $objectManager->get(\Windcave\Payments\Helper\PxPayIFrame\UrlCreator::class);
        $this->_apiCommonHelper = $objectManager->get(\Windcave\Payments\Model\Api\ApiCommonHelper::class);
     
        $this->_quoteFactory = $quoteFactory;
        $this->_logger = $objectManager->get(\Windcave\Payments\Logger\DpsLogger::class);
        
        $this->_logger->info(__METHOD__);
    }

    /**
     * Creates the PxPay URL for the logged-in customer
     *
     * @param string $quoteId
     * @param \Magento\Quote\Api\Data\PaymentInterface $method
     * @return string
     * @throws \Magento\Framework\Exception\State\InvalidTransitionException
     */
    public function createUrlForCustomer($quoteId, \Magento\Quote\Api\Data\PaymentInterface $method)
    {
        $this->_logger->info(__METHOD__. " quoteId:{$quoteId}");

        /** @var \Magento\Quote\Model\Quote $quote */
        $quote = $this->_apiCommonHelper->setPaymentForLoggedinCustomer($quoteId, $method);
        
        return $this->_createUrl($quote, null);
    }
    
    /**
     * Creates the PxPay URL for the Guest checkout
     *
     * @param string $cartId
     * @param string $email
     * @param \Magento\Quote\Api\Data\PaymentInterface $method
     * @return string
     * @throws \Magento\Framework\Exception\State\InvalidTransitionException
     */
    public function createUrlForGuest($cartId, $email, \Magento\Quote\Api\Data\PaymentInterface $method)
    {
        $this->_logger->info(__METHOD__. " cartId:{$cartId}");

        /** @var \Magento\Quote\Model\Quote $quote */
        $quote = $this->_apiCommonHelper->setPaymentForGuest($cartId, $email, $method);
        
        return $this->_createUrl($quote, null);
    }
    
    /**
     * Creates the PxPay URL for the order
     *
     * @param \Magento\Sales\Model\Order $order
     * @return string
     * @throws \Magento\Framework\Exception\State\InvalidTransitionException
     */
    public function createUrl(\Magento\Sales\Model\Order $order)
    {
        $this->_logger->info(__METHOD__. " orderId:{$order->getEntityId()} quoteId:{$order->getQuoteId()}");

        $quote = $this->_quoteFactory->create()->load($order->getQuoteId());
        $orderId = $order->getRealOrderId();
        return $this->_createUrl($quote, $orderId);
    }

    /**
     * Constructs the PxPay URL
     *
     * @param \Magento\Quote\Model\Quote $quote
     * @param string $orderId
     * @return string
     * @throws \Magento\Framework\Exception\State\InvalidTransitionException
     */
    private function _createUrl(\Magento\Quote\Model\Quote $quote, $orderId)
    {
        // Create pxpay redirect url.
        $url = $this->_pxpayUrlCreator->createUrl($quote, $orderId);
        if (!isset($url) || empty($url)) {
            $quoteId = $quote->getId();
            $this->_logger->critical(__METHOD__ . " Failed to create transaction quoteId:{$quoteId}");
            throw new InvalidTransitionException(__('Failed to create transaction.'));
        }
        
        $this->_logger->info(__METHOD__. " redirectUrl:{$url}");
        return $url;
    }
}
