<?php
namespace Windcave\Payments\Helper\PxFusion;

use \Magento\Framework\App\Config\ScopeConfigInterface;
use \Magento\Framework\App\Helper\AbstractHelper;
use \Magento\Framework\App\Helper\Context;
use \Magento\Framework\Module\ModuleListInterface;

class Configuration extends AbstractHelper
{
    private const PXFUSION_PATH = "payment/windcave_pxfusion/";
    private const MODULE_NAME = "Windcave_Payments";

    /**
     * @var \Magento\Framework\Module\ModuleListInterface
     */
    private $_moduleList;

    /**
     * @var \Magento\Framework\App\ProductMetadataInterface
     */
    private $_productMetadata;
    
    /**
     * Constructor
     *
     * @param \Magento\Framework\App\Helper\Context $context
     */
    public function __construct(Context $context)
    {
        parent::__construct($context);
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $this->_productMetadata = $objectManager->get(\Magento\Framework\App\ProductMetadataInterface::class);
        $this->_moduleList = $objectManager->get(\Magento\Framework\Module\ModuleListInterface::class);
        $this->_logger = $objectManager->get(\Windcave\Payments\Logger\DpsLogger::class);
    }

    /**
     * Returns the module version
     *
     * @return string
     */
    public function getModuleVersion()
    {
        $version = "unknown";
        if ($this->_productMetadata != null) {
            $version = $this->_productMetadata->getVersion();
        }

        if ($this->_moduleList == null) {
            return "M2:" . $version. " ext:unknown";
        }
        return "M2:" . $version . " ext:" . $this->_moduleList->getOne(self::MODULE_NAME)['setup_version'];
    }
    
    /**
     * Verifies PxFusion configuration
     *
     * @param string|null $storeId
     * @return bool
     */
    public function isValidForPxFusion($storeId = null)
    {
        $success = true;
        $len = !empty($this->getUserName($storeId)) ? strlen($this->getUserName($storeId)) : 0;
        if ($len < 1 || $len > 32) {
            $this->_logger->warn(__METHOD__ . " Invalid PxFusion Username");
            $success = false;
        }

        $len = !empty($this->getPassword($storeId)) ? strlen($this->getPassword($storeId)) : 0;
        if ($len < 1 || $len > 64) {
            $this->_logger->warn(__METHOD__ . " Invalid PxFusion Key");
            $success = false;
        }

        if (filter_var($this->getWsdl($storeId), FILTER_VALIDATE_URL) === false) {
            $this->_logger->warn(__METHOD__ . " Invalid PxFusion WSDL URL");
            $success = false;
        }

        if (filter_var($this->getPostUrl($storeId), FILTER_VALIDATE_URL) === false) {
            $this->_logger->warn(__METHOD__ . " Invalid PxFusion Post URL");
            $success = false;
        }
        return $success;
    }

    /**
     * Verifies PxPost configuration
     *
     * @param string|null $storeId
     * @return bool
     */
    public function isValidForPxPost($storeId = null)
    {
        $success = true;
        $len = !empty($this->getPxPostUsername($storeId)) ? strlen($this->getPxPostUsername($storeId)) : 0;
        if ($len < 1 || $len > 27) {
            $this->_logger->warn(__METHOD__ . " Invalid PxPost Username");
            $success = false;
        }

        $len = !empty($this->getPxPostPassword($storeId)) ? strlen($this->getPxPostPassword($storeId)) : 0;
        if ($len < 1) {
            $this->_logger->warn(__METHOD__ . " Invalid PxPost password");
            $success = false;
        }

        if (filter_var($this->getPxPostUrl($storeId), FILTER_VALIDATE_URL) === false) {
            $this->_logger->warn(__METHOD__ . " Invalid PxPost URL");
            $success = false;
        }

        return $success;
    }

    /**
     * Returns folder for lock files
     *
     * @param string|null $storeId
     * @return string
     */
    public function getLocksFolder($storeId = null)
    {
        return $this->_getStoreConfig("locksFolder", $storeId);
    }

    /**
     * Returns redirect details for redirect on failure
     *
     * @param string|null $storeId
     * @return array
     */
    public function getRedirectOnErrorDetails($storeId = null)
    {
        $redirectValue = $this->_getStoreConfig("redirectonerror", $storeId);
        if ($redirectValue == \Windcave\Payments\Model\Config\Source\RedirectOnErrorOptions::PAYMENT_INFO) {
            return [ 'url' => 'checkout', 'params' => [ '_fragment' => 'payment' ] ];
        }
        return [ 'url' => 'checkout/cart', 'params' => [] ];
    }

    /**
     * Returns where to redirect on payment failure
     *
     * @param string|null $storeId
     * @return int
     */
    public function getRedirectOnErrorMode($storeId = null)
    {
        $redirectValue = $this->_getStoreConfig("redirectonerror", $storeId);
        if ($redirectValue == \Windcave\Payments\Model\Config\Source\RedirectOnErrorOptions::PAYMENT_INFO) {
            return 1;
        }
        return 0;
    }

    /**
     * Returns the title for "Place Order" button
     *
     * @param string|null $storeId
     * @return string
     */
    public function getPlaceOrderButtonTitle($storeId = null)
    {
        return $this->_getStoreConfig("placeOrderButtonTitle", $storeId);
    }

    /**
     * Returns whether PxFusion should be allowed as a payment option
     *
     * @param string|null $storeId
     * @return bool
     */
    public function getEnabled($storeId = null)
    {
        return filter_var($this->_getStoreConfig("active", $storeId), FILTER_VALIDATE_BOOLEAN);
    }
    
    /**
     * Returns the PxFusion Username
     *
     * @param string|null $storeId
     * @return string
     */
    public function getUserName($storeId = null)
    {
        return $this->_getStoreConfig("username", $storeId);
    }
    
    /**
     * Returns the PxFusion Password/Key
     *
     * @param string|null $storeId
     * @return string
     */
    public function getPassword($storeId = null)
    {
        return $this->_getStoreConfig("password", $storeId, true);
    }
    
    /**
     * Returns the PxPost URL
     *
     * @param string|null $storeId
     * @return string
     */
    public function getPostUrl($storeId = null)
    {
        return $this->_getStoreConfig("postUrl", $storeId, false);
    }
   
    /**
     * Returns the PxFusion WSDL URL
     *
     * @param string|null $storeId
     * @return string
     */
    public function getWsdl($storeId = null)
    {
        return $this->_getStoreConfig("wsdl", $storeId);
    }
    
    /**
     * Returns the payment type
     *
     * @param string|null $storeId
     * @return string
     */
    public function getPaymentType($storeId = null)
    {
        return (string)$this->_getStoreConfig("paymenttype", $storeId);
    }
    
    /**
     * Returns the PxPost Username
     *
     * @param string|null $storeId
     * @return string
     */
    public function getPxPostUsername($storeId = null)
    {
        return $this->_getStoreConfig("pxpostusername", $storeId);
    }
    
    /**
     * Returns the PxPost Password
     *
     * @param string|null $storeId
     * @return string
     */
    public function getPxPostPassword($storeId = null)
    {
        return $this->_getStoreConfig("pxpostpassword", $storeId, true);
    }
    
    /**
     * Returns the PxPost URL
     *
     * @param string|null $storeId
     * @return string
     */
    public function getPxPostUrl($storeId = null)
    {
        return $this->_getStoreConfig("pxposturl", $storeId);
    }

    /**
     * Returns whether to allow storing cards for future rebilling
     *
     * @param string|null $storeId
     * @return bool
     */
    public function getAllowRebill($storeId = null)
    {
        return filter_var($this->_getStoreConfig("allowRebill", $storeId), FILTER_VALIDATE_BOOLEAN);
    }

    /**
     * Returns whether to require cardholder name input
     *
     * @param string|null $storeId
     * @return bool
     */
    public function getRequireCardholderName($storeId = null)
    {
        return filter_var($this->_getStoreConfig("requireCardholderName", $storeId), FILTER_VALIDATE_BOOLEAN);
    }

    /**
     * Returns whether to require CVC on rebilling
     *
     * @param string|null $storeId
     * @return bool
     */
    public function getRequireCvcForRebilling($storeId = null)
    {
        return filter_var($this->_getStoreConfig("requireCvcForRebilling", $storeId), FILTER_VALIDATE_BOOLEAN);
    }

    /**
     * Returns whether to show or hide the CVC guide
     *
     * @param string|null $storeId
     * @return bool
     */
    public function getShowCvcGuide($storeId = null)
    {
        return filter_var($this->_getStoreConfig("showCvcGuide", $storeId), FILTER_VALIDATE_BOOLEAN);
    }

    /**
     * Returns the new auth order status
     *
     * @param string|null $storeId
     * @return string
     */
    public function getNewAuthOrderStatus($storeId = null)
    {
        return $this->_getStoreConfig("auth_order_status", $storeId);
    }

    /**
     * Returns PxFusion store configuration value by name
     *
     * @param string $configName
     * @param string|null $storeId
     * @param bool $isSensitiveData
     * @return string
     */
    private function _getStoreConfig($configName, $storeId = null, $isSensitiveData = false)
    {
        $this->_logger->info(__METHOD__. " storeId:{$storeId}");
    
        $value = $this->scopeConfig->getValue(
            self::PXFUSION_PATH . $configName,
            \Magento\Store\Model\ScopeInterface::SCOPE_STORE,
            $storeId
        );
    
        if (!$isSensitiveData) {
            $this->_logger->info(__METHOD__ . " storeId:{$storeId} configName:{$configName} value:{$value}");
        } else {
            $this->_logger->info(__METHOD__ . " storeId:{$storeId} configName:{$configName} value:*****");
        }
        return $value;
    }
}
