<?php

namespace Windcave\Payments\Helper;

use \Magento\Framework\Filesystem\Driver\File as FileDriver;

class FileLock
{
    /**
     * @var string
     */
    private $_fileName;

    /**
     * @var string
     */
    private $_fileHandle;

    /**
     * @var \Magento\Framework\Filesystem\Driver\File
     */
    private $_fs;

    /**
     * Constructor
     *
     * @param string $lockName
     * @param string $lockPath
     */
    public function __construct($lockName, $lockPath = null)
    {
        $this->_fs = new FileDriver(false);
        $lockPath = $lockPath ?: sys_get_temp_dir();

        $isDirExists = false;
        try {
            $isDirExists = $this->_fs->isDirectory($lockPath);
        } catch (\Exception $ex) {
            $isDirExists = false;
        }
        if (!$isDirExists) {
            $this->_fs->createDirectory($lockPath);
        }

        if (!$this->_fs->isWritable($lockPath)) {
            throw new \Magento\Framework\Exception\FileSystemException(
                sprintf('The directory "%s" is not writable.', $lockPath)
            );
        }

        $this->_fileName = sprintf(
            '%s/px.%s.%s.lock',
            $lockPath,
            preg_replace('/[^a-z0-9\._-]+/i', '-', $lockName),
            hash('sha256', $lockName)
        );
    }

    /**
     * Wrapper for DriverInfo call that makes sure it doesn't throw
     *
     * @param string $fileName
     * @param string $mode
     * @return resource
     */
    private function tryOpenFile($fileName, $mode)
    {
        try {
            return $this->_fs->fileOpen($fileName, $mode);
        } catch (\Exception $ex) {
            return null;
        }
    }

    /**
     * Wrapper for DriverInfo call that makes sure it doesn't throw
     *
     * @param resource $resource
     * @param int $lockMode
     * @return bool
     */
    private function tryLockFile($resource, $lockMode)
    {
        try {
            return $this->_fs->fileLock($resource, $lockMode);
        } catch (\Exception $ex) {
            return false;
        }
    }

    /**
     * Attempts to get the lock
     *
     * @param bool $isBlocking
     * @return bool
     */
    public function tryLock($isBlocking = false)
    {
        if ($this->_fileHandle) {
            return true;
        }

        $error = null;

        set_error_handler(function ($errno, $msg) use (&$error) {
            $error = $msg;
        });

        $this->_fileHandle = $this->tryOpenFile($this->_fileName, 'r');
        if (!$this->_fileHandle) {
            $this->_fileHandle = $this->tryOpenFile($this->_fileName, 'x');
            if ($this->_fileHandle) {
                $this->_fs->changePermissions($this->_fileName, 0444);
            } else {
                $this->_fileHandle = $this->tryOpenFile($this->_fileName, 'r');
                if (!$this->_fileHandle) {
                    usleep(100);
                    $this->_fileHandle = $this->tryOpenFile($this->_fileName, 'r');
                }
            }
        }
        restore_error_handler();

        if (!$this->_fileHandle) {
            throw new \Magento\Framework\Exception\FileSystemException($error);
        }

        // On Windows, even if PHP doc says the contrary, LOCK_NB works, see https://bugs.php.net/54129
        if (!$this->tryLockFile($this->_fileHandle, LOCK_EX | ($isBlocking ? 0 : LOCK_NB))) {
            $this->_fs->fileClose($this->_fileHandle);
            $this->_fileHandle = null;

            return false;
        }

        return true;
    }

    /**
     * Releases the lock
     */
    public function release()
    {
        if ($this->_fileHandle) {
            $this->tryLockFile($this->_fileHandle, LOCK_UN | LOCK_NB);
            $this->_fs->fileClose($this->_fileHandle);
            $this->_fileHandle = null;
        }
    }
}
