<?php
namespace Windcave\Payments\Helper;

use \Magento\Framework\App\Helper\AbstractHelper;
use \Magento\Framework\App\Helper\Context;

class Communication extends AbstractHelper
{
    private const SENSITIVE_FIELDS = [
        "PxPayKey",
        "PostPassword"
    ];

    /**
     * @var \Windcave\Payments\Helper\PaymentUtil
     */
    private $_paymentUtil;

    /**
     * @var \Windcave\Payments\Helper\Configuration
     */
    private $_configuration;

    /**
     * @var Magento\Customer\Model\Session
     * */
    private $_customerSession;

    /**
     * @var \Magento\Framework\HTTP\Client\Curl
     */
    private $_curlClient;

    /**
     * Constructor
     *
     * @param \Magento\Framework\App\Helper\Context $context
     */
    public function __construct(Context $context)
    {
        parent::__construct($context);
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $this->_logger = $objectManager->get(\Windcave\Payments\Logger\DpsLogger::class);
        $this->_configuration = $objectManager->get(\Windcave\Payments\Helper\Configuration::class);
        $this->_paymentUtil = $objectManager->get(\Windcave\Payments\Helper\PaymentUtil::class);
        $this->_customerSession = $objectManager->get(\Magento\Customer\Model\Session::class);
        $this->_curlClient = $objectManager->get(\Magento\Framework\HTTP\Client\Curl::class);
        $this->_logger->info(__METHOD__);
    }

    /**
     * Generates PxPay2 session and returns HPP URL
     *
     * @param \Magento\Framework\DataObject $requestData
     * @param string|null $storeId
     * @return string|bool
     */
    public function getPxPay2Page($requestData, $storeId = null)
    {
        $this->_logger->info(__METHOD__);
        $requestXml = $this->_buildPxPay2Request($requestData);
        $url = $this->_configuration->getPxPayUrl($storeId);
        return $this->_sendRequest($requestXml, $url);
    }

    /**
     * Generates and sends PxPay transaction status request
     *
     * @param string $userId
     * @param string $token
     * @param string|null $storeId
     */
    public function getTransactionStatus($userId, $token, $storeId = null)
    {
        $this->_logger->info(__METHOD__ . " pxPayUserId:{$userId} storeId:{$storeId}");
        $requestXml = $this->_buildProcessResponseRequest($userId, $token);
        
        $pxPayUrl = $this->_configuration->getPxPayUrl($storeId);
        $responseXml = $this->_sendRequest($requestXml, $pxPayUrl);
        
        $this->_logger->info(__METHOD__ . " responseXml:" . $responseXml);
        return $responseXml;
    }

    /**
     * Generates and sends PxPost Refund request
     *
     * @param float $amount
     * @param string $currency
     * @param string $dpsTxnRef
     * @param string $storeId
     * @return string|bool
     */
    public function refund($amount, $currency, $dpsTxnRef, $storeId)
    {
        $this->_logger->info(
            __METHOD__ . " amount:{$amount} currency:{$currency} " .
            "dpsTxnRef:{$dpsTxnRef} storeId:{$storeId}"
        );
        $requestXml = $this->_buildRefundRequestXml($amount, $currency, $dpsTxnRef, $storeId);
        $url = $this->_configuration->getPxPostUrl($storeId);
        
        return $this->_sendRequest($requestXml, $url);
    }

    /**
     * Generates and sends PxPost Complete/Capture request
     *
     * @param float $amount
     * @param string $currency
     * @param string $dpsTxnRef
     * @param string $storeId
     * @return string|bool
     */
    public function complete($amount, $currency, $dpsTxnRef, $storeId)
    {
        $this->_logger->info(
            __METHOD__ . " amount:{$amount} currency:{$currency} " .
            "dpsTxnRef:{$dpsTxnRef} storeId:{$storeId}"
        );
        $requestXml = $this->_buildCompleteRequestXml($amount, $currency, $dpsTxnRef, $storeId);
        $url = $this->_configuration->getPxPostUrl($storeId);
        
        return $this->_sendRequest($requestXml, $url);
    }

    /**
     * Generates and sends PxPost Void request
     *
     * @param string $dpsTxnRef
     * @param string $storeId
     * @return string|null
     */
    public function void($dpsTxnRef, $storeId)
    {
        $this->_logger->info(__METHOD__ . " dpsTxnRef:{$dpsTxnRef} storeId:{$storeId}");
        $requestXml = $this->_buildVoidRequestXml($dpsTxnRef, $storeId);
        $url = $this->_configuration->getPxPostUrl($storeId);
        
        return $this->_sendRequest($requestXml, $url);
    }
    
    /**
     * Builds PxPay2 Generate Session request
     *
     * @param \Magento\Framework\DataObject $requestData
     * @param string|null $storeId
     * @return \SimpleXMLElement
     */
    private function _buildPxPay2Request($requestData, $storeId = null)
    {
        $this->_logger->info(__METHOD__);
        $userId = $this->_configuration->getPxPayUserId($storeId);
        $pxPayKey = $this->_configuration->getPxPayKey($storeId);
        
        $urlFail = $this->_getUrl('pxpay2/pxpay2/redirectFailure', ['_secure' => true]);
        $urlSuccess = $this->_getUrl('pxpay2/pxpay2/redirectSuccess', ['_secure' => true]);
        
        $amount = $requestData->getAmount();
        $currency = $requestData->getCurrency();
        $formattedAmount = $this->_paymentUtil->formatCurrency($amount, $currency);
        
        $requestObject = new \SimpleXMLElement("<GenerateRequest></GenerateRequest>");
        $requestObject->addChild("PxPayUserId", $userId);
        $requestObject->addChild("PxPayKey", $pxPayKey);
        $requestObject->addChild("TxnType", $requestData->getTransactionType());
        $requestObject->addChild("MerchantReference", $requestData->getOrderIncrementId());
        $requestObject->addChild("TxnId", $requestData->getTxnId());
        $requestObject->addChild("AmountInput", $formattedAmount);
        $requestObject->addChild("CurrencyInput", $currency);
        $requestObject->addChild("UrlFail", $urlFail);
        $requestObject->addChild("UrlSuccess", $urlSuccess);
        $requestObject->addChild("ClientVersion", $this->_configuration->getModuleVersion());
        
        if ($requestData->getForceA2A()) {
            $requestObject->addChild("ForcePaymentMethod", "Account2Account");
        }

        // field max length: https://www.windcave.com/developer-e-commerce-hosted-pxpay
        $addNonEmptyValue = function ($name, $value, $maxLength) use (&$requestObject) {
            if (isset($value) && $value) {
                $requestObject->addChild(
                    $name,
                    // phpcs:ignore Magento2.Functions.DiscouragedFunction
                    substr(htmlspecialchars($value, ENT_COMPAT | ENT_XML1), 0, $maxLength)
                );
            }
        };
        
        $dpsBillingId = $requestData->getDpsBillingId();
        if (!empty($dpsBillingId)) {
            $requestObject->addChild("DpsBillingId", $dpsBillingId);
        } else {
            if ($this->_customerSession->isLoggedIn()) {
                $addNonEmptyValue("EnableAddBillCard", $requestData->getEnableAddBillCard(), 1);
            }
        }
        
        $customerInfo = $requestData->getCustomerInfo();
        $addNonEmptyValue("TxnData1", $customerInfo->getName(), 255);
        $addNonEmptyValue("TxnData2", $customerInfo->getPhoneNumber(), 255);
        $addNonEmptyValue("TxnData3", $customerInfo->getAddress(), 255);
        
        $addNonEmptyValue("EmailAddress", $customerInfo->getEmail(), 255);
        $addNonEmptyValue("PhoneNumber", $customerInfo->getPhoneNumber(), 10);
        $addNonEmptyValue("AccountInfo", $customerInfo->getId(), 128);
        
        $requestXml = $requestObject->asXML();
        
        $this->_logger->info(__METHOD__ . " request: {$this->_obscureSensitiveFields($requestObject)}");
        return $requestXml;
    }

    /**
     * Builds PxPay ProcessResponse XML request
     *
     * @param string $userId
     * @param string $token
     * @return \SimpleXMLElement
     */
    private function _buildProcessResponseRequest($userId, $token)
    {
        $this->_logger->info(__METHOD__ . " pxPayUserId:{$userId} token:{$token}");
        $pxPayKey = "";
        if ($userId == $this->_configuration->getPxPayUserId()) {
            $pxPayKey = $this->_configuration->getPxPayKey();
        }
        
        $requestObject = new \SimpleXMLElement("<ProcessResponse></ProcessResponse>");
        $requestObject->addChild("PxPayUserId", $userId);
        $requestObject->addChild("PxPayKey", $pxPayKey);
        $requestObject->addChild("Response", $token);
        $requestObject->addChild("ClientVersion", $this->_configuration->getModuleVersion());
        
        $requestXml = $requestObject->asXML();
        
        $this->_logger->info(__METHOD__ . " request: {$this->_obscureSensitiveFields($requestObject)}");
        
        return $requestXml;
    }

    /**
     * Sends the XML request
     *
     * @param \SimpleXMLRequest $requestXml
     * @param string $postUrl
     * @return string|bool
     */
    private function _sendRequest($requestXml, $postUrl)
    {
        $this->_logger->info(__METHOD__ . " postUrl: {$postUrl}");

        $this->_curlClient->addHeader("Content-Type", "application/xml");
        $this->_curlClient->setTimeout(20);

        try {
            $this->_curlClient->post($postUrl, $requestXml);
        } catch (\Exception $ex) {
            $errorMessage = " Error:" . $ex->getMessage();
            $this->_logger->critical(__METHOD__ . $errorMessage);
        }

        $response = $this->_curlClient->getBody();
        $httpcode = $this->_curlClient->getStatus();
        if ($httpcode && substr($httpcode, 0, 2) != "20") {
            $errorMessage = " HTTP CODE: {$httpcode} for URL: {$postUrl}";
            $this->_logger->critical(__METHOD__ . $errorMessage);
        }
        
        $this->_logger->info(__METHOD__ . " response from PxPay2:" . $response);
        
        return $response;
    }

    /**
     * Builds PxPost Refund XML request
     *
     * @param float $amount
     * @param string $currency
     * @param string $dpsTxnRef
     * @param string $storeId
     * @return \SimpleXMLElement
     */
    private function _buildRefundRequestXml($amount, $currency, $dpsTxnRef, $storeId = null)
    {
        $this->_logger->info(__METHOD__);
        return $this->_buildPxPostRequestXml($amount, $currency, $dpsTxnRef, "Refund", $storeId);
    }

    /**
     * Builds PxPost Complete/Capture XML request
     *
     * @param float $amount
     * @param string $currency
     * @param string $dpsTxnRef
     * @param string $storeId
     * @return \SimpleXMLElement
     */
    private function _buildCompleteRequestXml($amount, $currency, $dpsTxnRef, $storeId = null)
    {
        $this->_logger->info(__METHOD__);
        return $this->_buildPxPostRequestXml($amount, $currency, $dpsTxnRef, "Complete", $storeId);
    }

    /**
     * Builds PxPost XML request
     *
     * @param float $amount
     * @param string $currency
     * @param string $dpsTxnRef
     * @param string $dpsTxnType
     * @param string $storeId
     * @return \SimpleXMLElement
     */
    private function _buildPxPostRequestXml($amount, $currency, $dpsTxnRef, $dpsTxnType, $storeId)
    {
        // <Txn>
        // <PostUsername>Pxpay_HubertFu</PostUsername>
        // <PostPassword>TestPassword</PostPassword>
        // <Amount>1.23</Amount>
        // <InputCurrency>NZD</InputCurrency>
        // <TxnType>Complete</TxnType>
        // <DpsTxnRef>000000600000005b</DpsTxnRef>
        // </Txn>
        $this->_logger->info(__METHOD__);
        $username = $this->_configuration->getPxPostUsername($storeId);
        $password = $this->_configuration->getPxPassword($storeId);
        
        $formattedAmount = $this->_paymentUtil->formatCurrency($amount, $currency);
        $requestObject = new \SimpleXMLElement("<Txn></Txn>");
        $requestObject->addChild("PostUsername", $username);
        $requestObject->addChild("PostPassword", $password);
        $requestObject->addChild("InputCurrency", $currency);
        $requestObject->addChild("Amount", $formattedAmount);
        $requestObject->addChild("DpsTxnRef", $dpsTxnRef);
        $requestObject->addChild("TxnType", $dpsTxnType);
        $requestObject->addChild("ClientVersion", $this->_configuration->getModuleVersion());

        $requestXml = $requestObject->asXML();
        
        $this->_logger->info(__METHOD__ . " request: {$this->_obscureSensitiveFields($requestObject)}");
        
        return $requestXml;
    }

    /**
     * Builds the Void request object
     *
     * @param string $dpsTxnRef
     * @param string|null $storeId
     * @return \SimpleXMLElement
     */
    private function _buildVoidRequestXml($dpsTxnRef, $storeId = null)
    {
        $this->_logger->info(__METHOD__);
        $username = $this->_configuration->getPxPostUsername($storeId);
        $password = $this->_configuration->getPxPassword($storeId);
        
        $requestObject = new \SimpleXMLElement("<Txn></Txn>");
        $requestObject->addChild("PostUsername", $username);
        $requestObject->addChild("PostPassword", $password);
        $requestObject->addChild("DpsTxnRef", $dpsTxnRef);
        $requestObject->addChild("TxnType", "Void");
        $requestObject->addChild("ClientVersion", $this->_configuration->getModuleVersion());

        $requestXml = $requestObject->asXML();
        
        $this->_logger->info(__METHOD__ . " request: {$this->_obscureSensitiveFields($requestObject)}");
        
        return $requestXml;
    }

    /**
     * Obscures the sensitive data
     *
     * @param \SimpleXMLElement $requestObject
     * @return string
     */
    private function _obscureSensitiveFields($requestObject)
    {
        foreach ($requestObject->children() as $child) {
            $name = $child->getName();
            if (in_array($name, self::SENSITIVE_FIELDS)) {
                $child[0] = "****";
            }
        }
        return $requestObject->asXML();
    }
}
