<?php
namespace Windcave\Payments\Helper\ApplePay;

use \Magento\Framework\App\Helper\AbstractHelper;
use \Magento\Framework\App\Helper\Context;

class Configuration extends AbstractHelper
{
    private const APPLEPAY_PATH = "payment/windcave_applepay/";
    private const MODULE_NAME = "Windcave_Payments";

    /**
     * Constructor
     *
     * @param \Magento\Framework\App\Helper\Context $context
     */
    public function __construct(Context $context)
    {
        parent::__construct($context);
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $this->_moduleList = $objectManager->get(\Magento\Framework\Module\ModuleListInterface::class);
        $this->_productMetadata = $objectManager->get(\Magento\Framework\App\ProductMetadataInterface::class);
        $this->_logger = $objectManager->get(\Windcave\Payments\Logger\DpsLogger::class);
    }

    /**
     * @var \Magento\Framework\Module\ModuleListInterface
     */
    private $_moduleList;

    /**
     * @var \Magento\Framework\App\ProductMetadataInterface
     */
    private $_productMetadata;

    /**
     * Returns the module version
     *
     * @return string
     */
    public function getModuleVersion()
    {
        $version = "unknown";
        if ($this->_productMetadata != null) {
            $version = $this->_productMetadata->getVersion();
        }

        if ($this->_moduleList == null) {
            return "M2:" . $version. " ext:unknown";
        }
        return "M2:" . $version . " ext:" . $this->_moduleList->getOne(self::MODULE_NAME)['setup_version'];
    }

    /**
     * Verifies Apple Pay configuration
     *
     * @param string|null $storeId
     * @return bool
     */
    public function isValidForApplePay($storeId = null)
    {
        $success = true;
        $len = !empty($this->getApiUserName($storeId)) ? strlen($this->getApiUserName($storeId)) : 0;
        if ($len < 1 || $len > 32) {
            $this->_logger->warn(__METHOD__ . " Invalid API Username");
            $success = false;
        }

        $len = !empty($this->getApiKey($storeId)) ? strlen($this->getApiKey($storeId)) : 0;
        if ($len < 1 || $len > 64) {
            $this->_logger->warn(__METHOD__ . " Invalid API Key");
            $success = false;
        }

        return $success;
    }

    /**
     * Verifies PxPost configuration
     *
     * @param string|null $storeId
     * @return bool
     */
    public function isValidForPxPost($storeId = null)
    {
        $success = true;
        $len = !empty($this->getPxPostUsername($storeId)) ? strlen($this->getPxPostUsername($storeId)) : 0;
        if ($len < 1 || $len > 27) {
            $this->_logger->warn(__METHOD__ . " Invalid PxPost Username");
            $success = false;
        }

        $len = !empty($this->getPxPostPassword($storeId)) ? strlen($this->getPxPostPassword($storeId)) : 0;
        if ($len < 1) {
            $this->_logger->warn(__METHOD__ . " Invalid PxPost password");
            $success = false;
        }

        if (filter_var($this->getPxPostUrl($storeId), FILTER_VALIDATE_URL) === false) {
            $this->_logger->warn(__METHOD__ . " Invalid PxPost URL");
            $success = false;
        }

        return $success;
    }

    /**
     * Returns whether the payment option is enabled
     *
     * @param string|null $storeId
     * @return bool
     */
    public function getEnabled($storeId = null)
    {
        return filter_var($this->_getApplePayStoreConfig("active", $storeId), FILTER_VALIDATE_BOOLEAN);
    }

    /**
     * Returns the Rest API username
     *
     * @param string|null $storeId
     * @return string
     */
    public function getApiUserName($storeId = null)
    {
        return $this->_getApplePayStoreConfig("apiusername", $storeId);
    }

    /**
     * Returns the Rest API Key
     *
     * @param string|null $storeId
     * @return string
     */
    public function getApiKey($storeId = null)
    {
        return $this->_getApplePayStoreConfig("apikey", $storeId, true);
    }

    /**
     * Returns the Rest API URL
     *
     * @param string|null $storeId
     * @return string
     */
    public function getApiUrl($storeId = null)
    {
        return $this->_getApplePayStoreConfig("apiurl", $storeId, true);
    }

    /**
     * Returns the PxPost username
     *
     * @param string|null $storeId
     * @return string
     */
    public function getPxPostUsername($storeId = null)
    {
        return $this->_getApplePayStoreConfig("pxpostusername", $storeId);
    }
    
    /**
     * Returns the PxPost password
     *
     * @param string|null $storeId
     * @return string
     */
    public function getPxPostPassword($storeId = null)
    {
        return $this->_getApplePayStoreConfig("pxpostpassword", $storeId, true);
    }
    
    /**
     * Returns the PxPost URL
     *
     * @param string|null $storeId
     * @return string
     */
    public function getPxPostUrl($storeId = null)
    {
        return $this->_getApplePayStoreConfig("pxposturl", $storeId);
    }

    /**
     * Returns the payment button type
     *
     * @param string|null $storeId
     * @return string
     */
    public function getPaymentButtonType($storeId = null)
    {
        return (string)$this->_getApplePayStoreConfig("buttontype", $storeId);
    }

    /**
     * Returns the payment button color
     *
     * @param string|null $storeId
     * @return string
     */
    public function getPaymentButtonColor($storeId = null)
    {
        return (string)$this->_getApplePayStoreConfig("buttoncolor", $storeId);
    }

    /**
     * Returns the merchant name
     *
     * @param string|null $storeId
     * @return string
     */
    public function getMerchantName($storeId = null)
    {
        return (string)$this->_getApplePayStoreConfig("merchname", $storeId);
    }
    
    /**
     * Returns supported card networks
     *
     * @param string|null $storeId
     * @return string
     */
    public function getSupportedNetworks($storeId = null)
    {
        return (string)$this->_getApplePayStoreConfig("supportednetwork1", $storeId);
    }

    /**
     * Returns the merchant identifier
     *
     * @param string|null $storeId
     * @return string
     */
    public function getMerchantIdentifier($storeId = null)
    {
        return (string)$this->_getApplePayStoreConfig("merchid", $storeId);
    }

    /**
     * Returns the payment type
     *
     * @param string|null $storeId
     * @return string
     */
    public function getPaymentType($storeId = null)
    {
        return (string)$this->_getApplePayStoreConfig("paymenttype", $storeId);
    }

    /**
     * Returns Apple Pay store config
     *
     * @param string $configName
     * @param string|null $storeId
     * @param bool $isSensitiveData
     */
    private function _getApplePayStoreConfig($configName, $storeId = null, $isSensitiveData = false)
    {
        $this->_logger->info(__METHOD__ . " storeId:" . $storeId);
        
        $value = $this->scopeConfig->getValue(
            self::APPLEPAY_PATH . $configName,
            \Magento\Store\Model\ScopeInterface::SCOPE_STORE,
            $storeId
        );
        
        if (!$isSensitiveData) {
            $this->_logger->info(__METHOD__ . " configName:{$configName} storeId:{$storeId} value:{$value}");
        } else {
            $this->_logger->info(__METHOD__ . " configName:{$configName} storeId:{$storeId} value:*****");
        }
        return $value;
    }

    /**
     * Returns redirect on error details
     *
     * @param string|null $storeId
     * @return array
     */
    public function getRedirectOnErrorDetails($storeId = null)
    {
        return [ 'url' => 'checkout/cart', 'params' => [] ];
    }
}
