<?php
namespace Windcave\Payments\Controller\PxPay2;

use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Action\Context;
use Magento\Framework\Registry;
use Magento\Sales\Model\Order\ShipmentFactory;
use Magento\Sales\Model\Order\Invoice;

use Windcave\Payments\Helper\FileLock;

/***
 * This is the CommonAction for old versions of Success/Fail actions.
 * Stays here to account for outstanding PxPay sessions.
 *
 * This class will be deprecated in the next major release.
 */
abstract class CommonActionCompat extends \Magento\Framework\App\Action\Action
{
    /**
     * @var \Magento\Quote\Model\QuoteManagement
     */
    private $_quoteManagement;

    /**
     * @var \Magento\Quote\Model\GuestCart\GuestCartManagement
     */
    private $_guestCartManagement;

    /**
     * @var \Magento\Checkout\Model\Session
     */
    private $_checkoutSession;

    /**
     * @var \Windcave\Payments\Helper\Communication
     */
    private $_communication;

    /**
     * @var \Windcave\Payments\Helper\Configuration
     */
    private $_configuration;

    /**
     * @var \Magento\Framework\Message\ManagerInterface
     */
    private $_messageManager;

    /**
     * @var \Windcave\Payments\Logger\DpsLogger
     */
    protected $_logger;

    /**
     * @var \Magento\Quote\Model\QuoteIdMaskFactory
     */
    private $_quoteIdMaskFactory;

    /**
     * @param Magento\Framework\App\Action\Context $context
     */
    public function __construct(Context $context)
    {
        parent::__construct($context);
        $this->_logger = $this->_objectManager->get(\Windcave\Payments\Logger\DpsLogger::class);
        $this->_communication = $this->_objectManager->get(\Windcave\Payments\Helper\Communication::class);
        $this->_configuration = $this->_objectManager->get(\Windcave\Payments\Helper\Configuration::class);
        
        $this->_quoteManagement = $this->_objectManager->get(\Magento\Quote\Model\QuoteManagement::class);
        $this->_guestCartManagement = $this->_objectManager->get(
            \Magento\Quote\Model\GuestCart\GuestCartManagement::class
        );
        $this->_checkoutSession = $this->_objectManager->get(\Magento\Checkout\Model\Session::class);
        $this->_messageManager = $this->_objectManager->get(\Magento\Framework\Message\ManagerInterface::class);
        $this->_quoteIdMaskFactory = $this->_objectManager->get(\Magento\Quote\Model\QuoteIdMaskFactory::class);
        
        $this->_logger->info(__METHOD__);
    }

    /**
     * Handles successful payment
     */
    public function success()
    {
        $this->_logger->info(__METHOD__);
        $this->_handlePaymentResponse(true);
    }

    /**
     * Handles failed payment
     */
    public function fail()
    {
        $this->_logger->info(__METHOD__);
        $this->_handlePaymentResponse(false);
    }

    /**
     * Handles the payment response
     *
     * @param bool $success
     */
    private function _handlePaymentResponse($success)
    {
        $pxPayUserId = $this->_configuration->getPxPayUserId();
        $token = $this->getRequest()->getParam('result');
        $this->_logger->info(__METHOD__ . " userId:{$pxPayUserId} token:{$token} success:{$success}");

        /**
         *
         * @var Windcave\Payments\Helper\FileLock
         */
        $lockHandler = null;
        try {
            $lockFolder = $this->_configuration->getLocksFolder();
            if (empty($lockFolder)) {
                $lockFolder = BP . "/var/locks";
            }

            $lockHandler = new FileLock($token, $lockFolder);
            if (!$lockHandler->tryLock(false)) {
                $action = $this->getRequest()->getActionName();
                $params = $this->getRequest()->getParams();
                $triedTime = 0;
                if (array_key_exists('TriedTime', $params)) {
                    $triedTime = $params['TriedTime'];
                }
                if ($triedTime > 40) { // 40 seconds should be enough
                    $this->_redirectToCartPageWithError("Failed to process the order, please contact support.");
                    $this->_logger->critical(
                        __METHOD__ . " lock timeout. userId:{$pxPayUserId} token:{$token} " .
                        "success:{$success} triedTime:{$triedTime}"
                    );
                    return;
                }
                
                $params['TriedTime'] = $triedTime + 1;
                
                $this->_logger->info(
                    __METHOD__ . " redirecting to self, wait for lock release. " .
                    "userId:{$pxPayUserId} token:{$token} success:{$success} triedTime:{$triedTime}"
                );
                // phpcs:ignore Magento2.Functions.DiscouragedFunction
                sleep(1); // wait for sometime about lock release
                return $this->_forward($action, null, null, $params);
            }
            
            $this->_handlePaymentResponseWithoutLock($success, $pxPayUserId, $token);
            $lockHandler->release();
        } catch (\Exception $e) {
            if (isset($lockHandler)) {
                $lockHandler->release();
            }
            
            $this->_logger->critical(__METHOD__ . "  " . "\n" . $e->getMessage() . $e->getTraceAsString());
            $this->_redirectToCartPageWithError("Failed to processing the order, please contact support.");
        }
    }

    /**
     * Handles the payment response
     *
     * @param bool $success
     * @param string $pxPayUserId
     * @param string $token
     */
    private function _handlePaymentResponseWithoutLock($success, $pxPayUserId, $token)
    {
        $this->_logger->info(__METHOD__ . " userId:{$pxPayUserId} token:{$token} success:{$success}");
        
        $cache = $this->_loadTransactionStatusFromCache($pxPayUserId, $token);
        $orderIncrementId = $cache->getOrderIncrementId();
        if (empty($orderIncrementId)) {
                
            $responseXmlElement = $this->_getTransactionStatus($pxPayUserId, $token);
            if (!$responseXmlElement) {
                return;
            }
            
            $orderIncrementId = (string)$responseXmlElement->MerchantReference;
            $dpsTxnRef = (string)$responseXmlElement->DpsTxnRef;
            
            $quote = $this->_loadQuote($orderIncrementId);
            if ($quote == null) {
                $error = "Failed to load quote from order: {$orderIncrementId}";
                $this->_logger->critical($error);
                $this->_redirectToCartPageWithError($error);
                return;
            }
            
            $this->_savePaymentResult($pxPayUserId, $token, $quote, $responseXmlElement);
            if (!$success) {
                $payment = $quote->getPayment();
                $this->_savePaymentInfoForFailedPayment($payment);
                
                $error = "Payment failed. Error: " . $responseXmlElement->ResponseText;
                $this->_logger->info($error);
                $this->_redirectToCartPageWithError($error);
                return;
            }
            
            return $this->_placeOrder($quote, $responseXmlElement);
        }
        
        if (!$success) {
            $responseXmlElement = $cache->getResponseXmlElement();
            
            $error = "Payment failed. Error: " . $responseXmlElement->ResponseText;
            $this->_logger->info($error);
            $this->_redirectToCartPageWithError($error);
            return;
        }
        
        $this->_redirect(
            "pxpay2/pxfusion/waitingQuote",
            [
                "_secure" => true,
                "triedtimes" => 0,
                "reservedorderid" => $orderIncrementId
            ]
        );
    }

    /**
     * Loads the quote by order Increment Id
     *
     * @param string $orderIncrementId
     */
    private function _loadQuote($orderIncrementId)
    {
        $this->_logger->info(__METHOD__ . " reserved_order_id:{$orderIncrementId}");
        
        $quoteManager = $this->_objectManager->create(\Magento\Quote\Model\Quote::class);
        /**
         * @var \Magento\Quote\Model\Quote $quote
         */
        $quote = $quoteManager->load($orderIncrementId, "reserved_order_id");
        
        if (!$quote->getId()) {
            $error = "Failed to load quote from order:{$orderIncrementId}";
            $this->_logger->critical($error);
            $this->_redirectToCartPageWithError($error);
            return null;
        }
        
        return $quote;
    }

    /**
     * Loads the transaction from cache
     *
     * @param string $pxPayUserId
     * @param string $token
     */
    private function _loadTransactionStatusFromCache($pxPayUserId, $token)
    {
        $this->_logger->info(__METHOD__ . " userId:{$pxPayUserId} token:{$token}");
        
        $paymentResultModel = $this->_objectManager->create(\Windcave\Payments\Model\PaymentResult::class);
        
        $paymentResultModelCollection = $paymentResultModel->getCollection()
            ->addFieldToFilter('token', $token)
            ->addFieldToFilter('user_name', $pxPayUserId);
        
        $paymentResultModelCollection->getSelect();
        
        $isProcessed = false;
        $dataBag = $this->_objectManager->create(\Magento\Framework\DataObject::class);

        $orderIncrementId = null;
        foreach ($paymentResultModelCollection as $item) {
            $orderIncrementId = $item->getReservedOrderId();
            $quoteId = $item->getQuoteId();
            $dataBag->setQuoteId($quoteId);
            $responseXmlElement = simplexml_load_string($item->getRawXml());
            $dataBag->setResponseXmlElement($responseXmlElement);
            $this->_logger->info(
                __METHOD__ . " userId:{$pxPayUserId} token:{$token} orderId:{$orderIncrementId} " .
                "quoteId:{$quoteId}"
            );
            break;
        }
        
        $dataBag->setOrderIncrementId($orderIncrementId);
        
        $this->_logger->info(
            __METHOD__ . " userId:{$pxPayUserId} token:{$token} " .
            "orderIncrementId:{$orderIncrementId}"
        );
        return $dataBag;
    }

    /**
     * Gets the boolean value from the array
     *
     * @param array $array
     * @param string $fieldName
     * @return bool
     */
    private function _getBoolValue($array, $fieldName)
    {
        if (!isset($array)) {
            return false;
        }
        if (!isset($array[$fieldName])) {
            return false;
        }

        return filter_var($array[$fieldName], FILTER_VALIDATE_BOOLEAN);
    }

    /**
     * Places the order
     *
     * @param \Magento\Quote\Model\Quote $quote
     * @param \SimpleXMLElement $responseXmlElement
     */
    private function _placeOrder(\Magento\Quote\Model\Quote $quote, $responseXmlElement)
    {
        $orderIncrementId = (string)$responseXmlElement->MerchantReference;
        $this->_logger->info(__METHOD__ . " orderIncrementId:{$orderIncrementId}");
        
        $quoteId = $quote->getId();
        $payment = $quote->getPayment();
        
        $info = $payment->getAdditionalInformation();
        $this->_logger->info(__METHOD__ . " info:" . var_export($info, true));
        
        $this->_savePaymentInfoForSuccessfulPayment($payment, $responseXmlElement);

        $isRegisteredCustomer = !empty($quote->getCustomerId());
        if ($isRegisteredCustomer) {
            // looks like $payment is copy by reference.
            // ensure the $payment data of order is exactly same with the quote.
            $quote->setPayment($payment);
            $this->_logger->info(__METHOD__ . " placing order for logged in customer. quoteId:{$quoteId}");
            // create order, and redirect to success page.
            $orderId = $this->_quoteManagement->placeOrder($quoteId);

            $enableAddBillCard =  $this->_getBoolValue($info, "EnableAddBillCard");
            if ($enableAddBillCard) {
                $this->_saveRebillToken($payment, $orderId, $quote->getCustomerId(), $responseXmlElement);
            }
        } else {
            // Guest checkout
            $cartId = $this->_quoteIdMaskFactory->create()->load($quoteId, 'quote_id')->getMaskedId();

            $this->_logger->info(__METHOD__ . " placing order for guest. quoteId:{$quoteId} cartId:{$cartId}");
            $orderId = $this->_guestCartManagement->placeOrder($cartId);
        }
        
        $this->_checkoutSession->setLoadInactive(false);
        $this->_checkoutSession->replaceQuote($this->_checkoutSession->getQuote()->save());
        
        $this->_logger->info(
            __METHOD__ . " placing order done " .
            "lastSuccessQuoteId:". $this->_checkoutSession->getLastSuccessQuoteId().
            " lastQuoteId:".$this->_checkoutSession->getLastQuoteId().
            " lastOrderId:".$this->_checkoutSession->getLastOrderId().
            " lastRealOrderId:" . $this->_checkoutSession->getLastRealOrderId()
        );
        
        $this->_redirect("checkout/onepage/success", [
            "_secure" => true
        ]);
    }

    /**
     * Saves the payment info for the successful payment
     *
     * @param \Magento\Sales\Model\Order\Payment $payment
     * @param \SimpleXMLElement $paymentResponseXmlElement
     */
    private function _savePaymentInfoForSuccessfulPayment($payment, $paymentResponseXmlElement)
    {
        $this->_logger->info(__METHOD__);
        $info = $payment->getAdditionalInformation();
        
        $info = $this->_clearPaymentParameters($info);
        
        $info["DpsTransactionType"] = (string)$paymentResponseXmlElement->TxnType;
        $info["DpsResponseText"] = (string)$paymentResponseXmlElement->ResponseText;
        $info["ReCo"] = (string)$paymentResponseXmlElement->ReCo;
        $info["DpsTransactionId"] = (string)$paymentResponseXmlElement->TxnId;
        $info["DpsTxnRef"] = (string)$paymentResponseXmlElement->DpsTxnRef;
        $info["CardName"] = (string)$paymentResponseXmlElement->CardName;
        $info["CardholderName"] = (string)$paymentResponseXmlElement->CardHolderName;
        $info["Currency"] = $payment->getOrder()->getOrderCurrencyCode();
        if ($this->_configuration->getAllowRebill()) {
            $info["DpsBillingId"] = (string)$paymentResponseXmlElement->DpsBillingId;
        }
        
        $payment->unsAdditionalInformation(); // ensure DpsBillingId is not saved to database.
        $payment->setAdditionalInformation($info);
        
        $info = $payment->getAdditionalInformation();
        $this->_logger->info(__METHOD__ . " info: ".var_export($info, true));
        $payment->save();
    }
    
    /**
     * Saves the payment info for the failed payment
     *
     * @param \Magento\Sales\Model\Order\Payment $payment
     */
    private function _savePaymentInfoForFailedPayment($payment)
    {
        $this->_logger->info(__METHOD__);
        $info = $payment->getAdditionalInformation();
    
        $info = $this->_clearPaymentParameters($info);

        $payment->unsAdditionalInformation(); // ensure DpsBillingId is not saved to database.
        $payment->setAdditionalInformation($info);
        $payment->save();
    }
    
    /**
     * Clear the payment parameters
     *
     * @param array $info
     * @return array
     */
    private function _clearPaymentParameters($info)
    {
        $this->_logger->info(__METHOD__);
        
        unset($info["cartId"]);
        unset($info["guestEmail"]);
        unset($info["UseSavedCard"]);
        unset($info["DpsBillingId"]);
        unset($info["EnableAddBillCard"]);
        unset($info["method_title"]);

        $this->_logger->info(__METHOD__ . " info: ".var_export($info, true));
        return $info;
    }

    /**
     * Saves the billing token against the customer
     *
     * @param \Magento\Sales\Model\Order\Payment $payment
     * @param string $orderId
     * @param string $customerId
     * @param \SimpleXMLElement $paymentResponseXmlElement
     */
    private function _saveRebillToken($payment, $orderId, $customerId, $paymentResponseXmlElement)
    {
        $this->_logger->info(__METHOD__." orderId:{$orderId}, customerId:{$customerId}");
        $storeManager = $this->_objectManager->get(\Magento\Store\Model\StoreManagerInterface::class);
        $storeId = $storeManager->getStore()->getId();
        $billingModel = $this->_objectManager->create(\Windcave\Payments\Model\BillingToken::class);
        $billingModel->setData(
            [
                "customer_id" => $customerId,
                "order_id" => $orderId,
                "store_id" => $storeId,
                "masked_card_number" => (string)$paymentResponseXmlElement->CardNumber,
                "cc_expiry_date" => (string)$paymentResponseXmlElement->DateExpiry,
                "dps_billing_id" => (string)$paymentResponseXmlElement->DpsBillingId
            ]
        );
        $billingModel->save();
    }

    /**
     * Saves the payment result in the Windcave Payment Result table
     *
     * @param string $pxpayUserId
     * @param string $token
     * @param \Magento\Quote\Model\Quote $quote
     * @param \SimpleXMLElement $paymentResponseXmlElement
     */
    private function _savePaymentResult(
        $pxpayUserId,
        $token,
        \Magento\Quote\Model\Quote $quote,
        $paymentResponseXmlElement
    ) {
        $this->_logger->info(__METHOD__ . " username:{$pxpayUserId}, token:{$token}");
        $payment = $quote->getPayment();
        $method = $payment->getMethod();
        
        $paymentResultModel = $this->_objectManager->create(\Windcave\Payments\Model\PaymentResult::class);
        $paymentResultModel->setData(
            [
                "dps_transaction_type" => (string)$paymentResponseXmlElement->TxnType,
                "dps_txn_ref" => (string)$paymentResponseXmlElement->DpsTxnRef,
                "method" => $method,
                "user_name" => $pxpayUserId,
                "token" => $token,
                "quote_id" => $quote->getId(),
                "reserved_order_id" => (string)$paymentResponseXmlElement->MerchantReference,
                "updated_time" => new \DateTime(),
                "raw_xml" => (string)$paymentResponseXmlElement->asXML()
            ]
        );
        
        $paymentResultModel->save();
        
        $this->_logger->info(__METHOD__ . " done");
    }

    /**
     * Queries the transaction status
     *
     * @param string $pxPayUserId
     * @param string $token
     * @return \SimpleXMLElement
     */
    private function _getTransactionStatus($pxPayUserId, $token)
    {
        $responseXml = $this->_communication->getTransactionStatus($pxPayUserId, $token);
        $responseXmlElement = simplexml_load_string($responseXml);
        if (!$responseXmlElement) { // defensive code. should never happen
            $this->_logger->critical(
                __METHOD__ . " userId:{$pxPayUserId} token:{$token} " .
                "response format is incorrect"
            );
            $this->_redirectToCartPageWithError("Failed to connect to Windcave. Please try again later.");
            return false;
        }
        
        return $responseXmlElement;
    }
    
    /**
     * Redirects to the cart page with an error message
     *
     * @param string $error
     */
    private function _redirectToCartPageWithError($error)
    {
        $this->_logger->info(__METHOD__ . " error:{$error}");
        
        $this->_messageManager->addErrorMessage($error);

        $redirectDetails = $this->_configuration->getRedirectOnErrorDetails();
        $this->_redirect($redirectDetails['url'], $redirectDetails['params']);
    }
}
