<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-seo
 * @version   2.10.12
 * @copyright Copyright (C) 2025 Mirasvit (https://mirasvit.com/)
 */



namespace Mirasvit\Seo\Plugin\Frontend;

use Magento\Framework\App\Action\Forward;
use Magento\Framework\App\ActionFactory;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\UrlInterface;
use Magento\Store\Model\StoreManagerInterface;
use Magento\UrlRewrite\Controller\Router;
use Magento\UrlRewrite\Model\UrlFinderInterface;
use Magento\UrlRewrite\Service\V1\Data\UrlRewrite;
use Mirasvit\Seo\Model\Config;
use Mirasvit\Seo\Helper\Redirect as RedirectHelper;

/**
 * @see \Magento\UrlRewrite\Controller\Router::match();
 */
class UrlRewriteRouterApplyTrailingSlashPlugin
{
    private $redirectHelper;

    private $config;

    private $urlFinder;

    private $storeManager;

    private $actionFactory;

    public function __construct(
        Config                $config,
        UrlFinderInterface    $urlFinder,
        StoreManagerInterface $storeManager,
        RedirectHelper        $redirectHelper,
        ActionFactory         $actionFactory
    ) {
        $this->config         = $config;
        $this->urlFinder      = $urlFinder;
        $this->storeManager   = $storeManager;
        $this->redirectHelper = $redirectHelper;
        $this->actionFactory  = $actionFactory;
    }

    public function aroundMatch(Router $subject, callable $proceed, RequestInterface $request)
    {
        if (strpos($request->getFullActionName(), 'checkout') !== false
            || $this->config->getTrailingSlash() == Config::TRAILING_SLASH_DISABLE
            || $request->getPathInfo() != $request->getOriginalPathInfo()
        ) {
            return $proceed($request);
        }

        $storeId = $this->storeManager->getStore()->getId();

        $rewriteUrl = trim($request->getPathInfo(), '/');

        if ($this->config->getTrailingSlash() == Config::TRAILING_SLASH) {
            $rewriteUrl .= '/';
        }

        $rewrite = $this->getRewrite($rewriteUrl, $storeId);

        if (!$this->checkIfRewriteIdExists($rewrite)) {
            if (substr($rewriteUrl, -1) != '/') {
                $rewriteUrl .= '/';
            } else {
                $rewriteUrl = trim($rewriteUrl, '/');
            }
            $rewrite = $this->getRewrite($rewriteUrl, $storeId);

            if ($this->checkIfRewriteIdExists($rewrite)) {
                $requestPath = $this->redirectHelper->getUrlWithCorrectEndSlash($rewrite->getTargetPath() ?? '');
                $requestPath = preg_replace('~\?.*~', '', $requestPath);

                $request->setAlias(
                    UrlInterface::REWRITE_REQUEST_PATH_ALIAS,
                    $rewrite->getRequestPath()
                );
                $request->setPathInfo($requestPath);

                return $this->actionFactory->create(
                    Forward::class
                );
            }
        }

        return $proceed($request);
    }

    private function checkIfRewriteIdExists(?UrlRewrite $rewrite): bool
    {
        return !empty($rewrite) && $rewrite->getUrlRewriteId();
    }

    private function getRewrite(string $requestPath, ?int $storeId): ?UrlRewrite
    {
        return $this->urlFinder->findOneByData(
            [
                UrlRewrite::REQUEST_PATH => ltrim($requestPath, '/'),
                UrlRewrite::STORE_ID => $storeId,
            ]
        );
    }
}
