<?php

namespace Overdose\InventoryShipping\Observer;

use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\Event\Observer as EventObserver;
use Magento\Framework\Exception\LocalizedException;
use Magento\InventorySalesApi\Api\Data\SalesEventExtensionInterface;
use Magento\InventorySalesApi\Api\Data\SalesEventInterface;
use Magento\InventorySalesApi\Api\Data\SalesEventInterfaceFactory;
use Magento\InventorySalesApi\Api\Data\SalesEventExtensionFactory;
use Magento\InventorySourceDeductionApi\Model\SourceDeductionRequestInterface;
use Magento\InventorySourceDeductionApi\Model\SourceDeductionServiceInterface;
use Magento\InventoryShipping\Model\SourceDeductionRequestsFromSourceSelectionFactory;
use Magento\Sales\Api\Data\InvoiceInterface;
use Magento\InventorySalesApi\Api\Data\ItemToSellInterfaceFactory;
use Magento\InventorySalesApi\Api\PlaceReservationsForSalesEventInterface;
use Magento\Sales\Model\Order;
use Magento\Sales\Model\Order\Invoice;
use Psr\Log\LoggerInterface;
use Overdose\InventoryShipping\Model\ResourceModel\GetReservationsForInvoice;
use Overdose\InventoryShipping\Model\ResourceModel\OrderSource\SaveOrderSource;
use Overdose\InventoryShipping\Model\GetSourceSelectionResultFromInvoice;

/**
 * Class source deduction mechanism
 * This class now deducts stock on invoice save instead of shipment save
 */
class SourceDeductionProcessor implements ObserverInterface
{
    /**
     * @var GetSourceSelectionResultFromInvoice
     */
    private GetSourceSelectionResultFromInvoice $getSourceSelectionResultFromInvoice;

    /**
     * @var SourceDeductionServiceInterface
     */
    private SourceDeductionServiceInterface $sourceDeductionService;

    /**
     * @var SourceDeductionRequestsFromSourceSelectionFactory
     */
    private SourceDeductionRequestsFromSourceSelectionFactory $sourceDeductionRequestsFromSourceSelectionFactory;

    /**
     * @var SalesEventInterfaceFactory
     */
    private SalesEventInterfaceFactory $salesEventFactory;

    /**
     * @var ItemToSellInterfaceFactory
     */
    private ItemToSellInterfaceFactory $itemToSellFactory;

    /**
     * @var PlaceReservationsForSalesEventInterface
     */
    private PlaceReservationsForSalesEventInterface $placeReservationsForSalesEvent;

    /**
     * @var SalesEventExtensionFactory;
     */
    private SalesEventExtensionFactory $salesEventExtensionFactory;
    /**
     * @var SaveOrderSource
     */
    private SaveOrderSource $saveOrderSource;
    /**
     * @var GetReservationsForInvoice
     */
    private GetReservationsForInvoice $getReservationsForInvoice;
    /**
     * @var LoggerInterface
     */
    private LoggerInterface $logger;

    /**
     * @param GetSourceSelectionResultFromInvoice $getSourceSelectionResultFromInvoice
     * @param SourceDeductionServiceInterface $sourceDeductionService
     * @param SourceDeductionRequestsFromSourceSelectionFactory $sourceDeductionRequestsFromSourceSelectionFactory
     * @param SalesEventInterfaceFactory $salesEventFactory
     * @param ItemToSellInterfaceFactory $itemToSellFactory
     * @param PlaceReservationsForSalesEventInterface $placeReservationsForSalesEvent
     * @param SalesEventExtensionFactory $salesEventExtensionFactory
     * @param SaveOrderSource $saveOrderSource
     * @param GetReservationsForInvoice $getReservationsForInvoice
     * @param LoggerInterface $logger
     */
    public function __construct(
        GetSourceSelectionResultFromInvoice $getSourceSelectionResultFromInvoice,
        SourceDeductionServiceInterface $sourceDeductionService,
        SourceDeductionRequestsFromSourceSelectionFactory $sourceDeductionRequestsFromSourceSelectionFactory,
        SalesEventInterfaceFactory $salesEventFactory,
        ItemToSellInterfaceFactory $itemToSellFactory,
        PlaceReservationsForSalesEventInterface $placeReservationsForSalesEvent,
        SalesEventExtensionFactory $salesEventExtensionFactory,
        SaveOrderSource $saveOrderSource,
        GetReservationsForInvoice $getReservationsForInvoice,
        LoggerInterface $logger
    ) {
        $this->getSourceSelectionResultFromInvoice = $getSourceSelectionResultFromInvoice;
        $this->sourceDeductionService = $sourceDeductionService;
        $this->sourceDeductionRequestsFromSourceSelectionFactory = $sourceDeductionRequestsFromSourceSelectionFactory;
        $this->salesEventFactory = $salesEventFactory;
        $this->itemToSellFactory = $itemToSellFactory;
        $this->placeReservationsForSalesEvent = $placeReservationsForSalesEvent;
        $this->salesEventExtensionFactory = $salesEventExtensionFactory;
        $this->saveOrderSource = $saveOrderSource;
        $this->getReservationsForInvoice = $getReservationsForInvoice;
        $this->logger = $logger;
    }

    /**
     * Process source deduction
     *
     * @param EventObserver $observer
     * @return void
     * @throws LocalizedException
     */
    public function execute(EventObserver $observer)
    {
        $this->logger->info('Start ' . __METHOD__);
        /** @var Order $order */
        $event = $observer->getEvent();
        /** @var Invoice $invoice */
        $invoice = $event->getInvoice();
        $this->processInvoice($invoice);
    }

    /**
     * @param Invoice $invoice
     */
    private function processInvoice(Invoice $invoice)
    {
        if (!$this->isValid($invoice)) {
            return;
        }
        $this->logger->info('Start invoice stock deductions/reservations for order id ' . $invoice->getOrderId());

        /** @var SalesEventExtensionInterface */
        $salesEventExtension = $this->salesEventExtensionFactory->create([
            'data' => ['objectIncrementId' => (string)$invoice->getOrder()->getIncrementId()]
        ]);

        /** @var SalesEventInterface $salesEvent */
        $salesEvent = $this->salesEventFactory->create([
            'type' => SalesEventInterface::EVENT_INVOICE_CREATED,
            'objectType' => SalesEventInterface::OBJECT_TYPE_ORDER,
            'objectId' => $invoice->getOrderId(),
        ]);
        $salesEvent->setExtensionAttributes($salesEventExtension);

        try {
            $sourceSelectionResult = $this->getSourceSelectionResultFromInvoice->execute($invoice);
            $sourceDeductionRequests = $this->sourceDeductionRequestsFromSourceSelectionFactory->create(
                $sourceSelectionResult,
                $salesEvent,
                (int)$invoice->getOrder()->getStore()->getWebsiteId()
            );
        } catch (\Exception $e) {
            //Ignore error
            $sourceDeductionRequests = [];
        }

        if (!empty($sourceDeductionRequests)) {
            $this->logger->info('Process invoice stock deductions/reservations for order id '
                . $invoice->getOrderId());
            foreach ($sourceDeductionRequests as $sourceDeductionRequest) {
                $this->saveSourceOrderItems($invoice, $sourceDeductionRequest);
                $this->sourceDeductionService->execute($sourceDeductionRequest);
                $this->placeCompensatingReservation($sourceDeductionRequest);
            }
            $invoice->setData('process_invoice_shipping', true);
        } else {
            $this->logger->info('Empty Source selection result for order id '. $invoice->getOrderId());
        }
        $this->logger->info('End ' . __METHOD__);
    }

    /**
     * Place compensating reservation after source deduction
     *
     * @param SourceDeductionRequestInterface $sourceDeductionRequest
     */
    private function placeCompensatingReservation(SourceDeductionRequestInterface $sourceDeductionRequest): void
    {
        $items = [];
        foreach ($sourceDeductionRequest->getItems() as $item) {
            $items[] = $this->itemToSellFactory->create([
                'sku' => $item->getSku(),
                'qty' => $item->getQty()
            ]);
        }
        $this->placeReservationsForSalesEvent->execute(
            $items,
            $sourceDeductionRequest->getSalesChannel(),
            $sourceDeductionRequest->getSalesEvent()
        );
    }

    /**
     * Is invoice valid.
     *
     * @param InvoiceInterface $invoice
     * @return bool
     */
    private function isValid(InvoiceInterface $invoice): bool
    {
        if ($invoice->getOrigData('entity_id')) {
            return false;
        }
        //Check for existing invoice reservations to prevent duplicate entries
        $existing = $this->getReservationsForInvoice->execute($invoice->getOrderId());
        if (count($existing) >= 1) {
            $this->logger->info('Ignore invoice stock deductions/reservations for order id '
                . $invoice->getOrderId());
            return false;
        }

        return true;
    }

    /**
     * Save source for order items
     *
     * @param $invoice
     * @param $sourceDeductionRequest
     * @return void
     */
    protected function saveSourceOrderItems($invoice, $sourceDeductionRequest)
    {
        $sourceCode = $sourceDeductionRequest->getSourceCode();
        foreach ($sourceDeductionRequest->getItems() as $item) {
            $this->saveOrderSource->execute(
                (int)$invoice->getOrderId(),
                $sourceCode,
                $item->getSku(),
                $item->getQty()
            );
        }
    }
}
