<?php

namespace Overdose\InStorePickup\Model\Service;

use Magento\Framework\Exception\LocalizedException;
use Magento\Store\Model\StoreManagerInterface;
use Magento\InventorySalesApi\Model\GetAssignedStockIdForWebsiteInterface;
use Magento\Inventory\Model\ResourceModel\StockSourceLink\CollectionFactory;
use Magento\Inventory\Model\ResourceModel\StockSourceLink\Collection;
use Magento\InventoryApi\Api\Data\StockSourceLinkInterface;
use Magento\Inventory\Model\ResourceModel\Source;

class SourcesService
{
    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var GetAssignedStockIdForWebsiteInterface
     */
    private $assignedStock;

    /**
     * @var CollectionFactory
     */
    private $collectionFactory;

    /**
     * SourcesService constructor.
     * @param StoreManagerInterface $storeManager
     * @param GetAssignedStockIdForWebsiteInterface $assignedStock
     * @param CollectionFactory $collectionFactory
     */
    public function __construct(
        StoreManagerInterface                 $storeManager,
        GetAssignedStockIdForWebsiteInterface $assignedStock,
        CollectionFactory                     $collectionFactory
    ) {
        $this->storeManager = $storeManager;
        $this->assignedStock = $assignedStock;
        $this->collectionFactory = $collectionFactory;
    }

    /**
     * Returns sources for a website
     * @param string|null $websiteCode
     * @return \Magento\Framework\DataObject[]
     * @throws LocalizedException
     */
    public function execute(string $websiteCode = null): array
    {
        if ($websiteCode === null) {
            $websiteCode = $this->storeManager->getWebsite()->getCode();
        }
        $stockId = $this->assignedStock->execute($websiteCode);

        $collection = $this->collectionFactory->create();
        $collection->addFieldToFilter('stock_id', $stockId);
        $this->joinInventorySource($collection);
        return $collection->getItems();
    }

    /**
     * @param Collection $collection
     * @return void
     */
    private function joinInventorySource(Collection $collection)
    {
        $collection->getSelect()->joinLeft(
            [
                'source_table' => $collection->getTable(
                    Source::TABLE_NAME_SOURCE
                ),
            ],
            '`main_table`.`source_code` = `source_table`.`source_code`'
        )->columns('source_table.enabled')
            ->where("`source_table`.`enabled` = '1'")
            ->order('main_table.priority ASC');
    }
}
