<?php

namespace Overdose\InStorePickup\Model\Checkout;

use Magento\Checkout\Model\Session;
use Magento\InventorySalesApi\Api\Data\SalesChannelInterface;
use Magento\InventorySalesApi\Api\StockResolverInterface;
use Magento\InventorySalesApi\Model\GetStockItemDataInterface;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Inventory\Model\ResourceModel\Source\CollectionFactory as SourceCollectionFactory;
use Magento\Inventory\Model\Source\Command\GetSourcesAssignedToStockOrderedByPriority;
use Magento\Inventory\Model\ResourceModel\SourceItem\CollectionFactory as SourceItemCollectionFactory;

class ConfigProvider
{
    /**
     * @var CartRepositoryInterface
     */
    private $cartRepository;

    /**
     * @var StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var StockResolverInterface
     */
    protected $stockResolver;

    /**
     * @var GetStockItemDataInterface
     */
    private $getStockItemData;

    /**
     * @var SourceCollectionFactory
     */
    protected $sourceCollectionFactory;

    /**
     * @var GetSourcesAssignedToStockOrderedByPriority
     */
    protected $getSourcesAssignedToStockOrderedByPriority;

    /**
     * @var SourceItemCollectionFactory
     */
    protected $sourceItemCollectionFactory;

    /**
     * @var Session
     */
    protected $checkoutSession;

    /**
     * @param CartRepositoryInterface $cartRepository
     * @param StoreManagerInterface $storeManager
     * @param StockResolverInterface $stockResolver
     * @param GetStockItemDataInterface $getStockItemData
     * @param SourceCollectionFactory $sourceCollectionFactory
     * @param SourceItemCollectionFactory $sourceItemCollectionFactory
     * @param Session $checkoutSession
     * @param GetSourcesAssignedToStockOrderedByPriority $getSourcesAssignedToStockOrderedByPriority
     */
    public function __construct(
        CartRepositoryInterface                    $cartRepository,
        StoreManagerInterface                      $storeManager,
        StockResolverInterface                     $stockResolver,
        GetStockItemDataInterface                  $getStockItemData,
        SourceCollectionFactory                    $sourceCollectionFactory,
        SourceItemCollectionFactory                $sourceItemCollectionFactory,
        Session                                    $checkoutSession,
        GetSourcesAssignedToStockOrderedByPriority $getSourcesAssignedToStockOrderedByPriority
    ) {
        $this->cartRepository = $cartRepository;
        $this->storeManager = $storeManager;
        $this->stockResolver = $stockResolver;
        $this->getStockItemData = $getStockItemData;
        $this->sourceCollectionFactory = $sourceCollectionFactory;
        $this->sourceItemCollectionFactory = $sourceItemCollectionFactory;
        $this->checkoutSession = $checkoutSession;
        $this->getSourcesAssignedToStockOrderedByPriority = $getSourcesAssignedToStockOrderedByPriority;
    }

    /**
     * @return array[]
     */
    public function getConfig()
    {
        return [
            'overdoseInStorePickup' => [
                'is_available_method' => $this->isAvailableMethod()
            ]
        ];
    }

    /**
     * @return bool
     */
    private function isAvailableMethod()
    {
        $quoteId = (int)$this->checkoutSession->getQuoteId();
        try {
            $listItemSku = [];
            $listInStorePickupSource = [];
            $listProductStockData = [];
            $website = $this->storeManager->getWebsite();
            $stock = $this->stockResolver->execute(SalesChannelInterface::TYPE_WEBSITE, $website->getCode());
            $stockId = (int)$stock->getStockId();
            $sourceList = $this->getSourcesAssignedToStockOrderedByPriority->execute($stockId);
            foreach ($sourceList as $source) {
                if ($source->getIsPickupLocationActive() && $source->isEnabled()) {
                    $listInStorePickupSource[] = $source->getSourceCode();
                }
            }
            $cart = $this->cartRepository->get($quoteId);
            $cartItems = $cart->getItems();
            foreach ($cartItems as $item) {
                $listItemSku[] = $item->getSku();
            }
            $sourceItemCollection = $this->sourceItemCollectionFactory
                ->create()
                ->addFieldToFilter('sku', ['in' => $listItemSku])
                ->addFieldToFilter('source_code', ['in' => $listInStorePickupSource]);
            foreach ($sourceItemCollection as $sourceItem) {
                if ((int)$sourceItem->getQuantity() > 0 && $sourceItem->getStatus() == 1) {
                    $listProductStockData[$sourceItem->getSourceCode()][$sourceItem->getSku()] = true;
                } else {
                    $listProductStockData[$sourceItem->getSourceCode()][$sourceItem->getSku()] = false;
                }
            }

            foreach ($listItemSku as $sku) {
                foreach ($listInStorePickupSource as $inStorePickupSource) {
                    if (isset($listProductStockData[$inStorePickupSource][$sku]) && $listProductStockData[$inStorePickupSource][$sku]) {
                        return true;
                    }
                }
                return false;
            }
        } catch (\Exception $e) {
            return false;
        }
        return true;
    }
}
