<?php
namespace Overdose\Catalog\Helper;

use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Helper\Context;
use Magento\InventorySalesApi\Api\Data\SalesChannelInterface;

/**
 * Class Stock
 * @package Overdose\Catalog\Helper
 */
class Stock extends AbstractHelper
{
    /**
     * @var \Overdose\Catalog\Model\ConfigProvider
     */
    private $configProvider;

    private $stockRegistry;

    public $productRepository;

    public $productSalable;

    public $stockResolver;


    public $storeManager;


    /**
     * Stock constructor.
     * @param Context $context
     * @param \Overdose\Catalog\Model\ConfigProvider $configProvider'
     */
    public function __construct(
        Context $context,
        \Overdose\Catalog\Model\ConfigProvider $configProvider,
        \Magento\CatalogInventory\Api\StockRegistryInterface $stockRegistry,
        \Magento\Catalog\Api\ProductRepositoryInterface $productRepository,
        \Magento\InventorySalesApi\Api\GetProductSalableQtyInterface $productSalable,
        \Magento\InventorySalesApi\Api\StockResolverInterface $stockResolver,
        \Magento\Store\Model\StoreManagerInterface $storeManager
    )
    {
        parent::__construct($context);
        $this->configProvider = $configProvider;
        $this->stockRegistry = $stockRegistry;
        $this->productRepository = $productRepository;
        $this->productSalable = $productSalable;
        $this->stockResolver = $stockResolver;
        $this->storeManager  = $storeManager;
    }

    /**
     * Get website code
     *
     * @return string|null
     */
    public function getWebsiteCode(): ?string
    {
        try {
            $websiteCode = $this->storeManager->getWebsite()->getCode();
        } catch (LocalizedException $localizedException) {
            $websiteCode = null;
            $this->logger->error($localizedException->getMessage());
        }
        return $websiteCode;
    }

    public function getSaleableStock($product_sku)
    {
        // $productSalable = $objectManager->get('\Magento\InventorySalesApi\Api\GetProductSalableQtyInterface');
        // $stockResolver = $objectManager->get('\Magento\InventorySalesApi\Api\StockResolverInterface');

        $stockId    = $this->stockResolver->execute(SalesChannelInterface::TYPE_WEBSITE, $this->getWebsiteCode())->getStockId();
        $stockQty   = $this->productSalable->execute($product_sku, $stockId);

        return $stockQty;
    }


    /**
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getInStockText()
    {
        return $this->configProvider->inStockText();
    }

    /**
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getLowStockThreshold()
    {
        return $this->configProvider->lowStockThreshold();
    }

    /**
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getLowStockText()
    {
        return $this->configProvider->lowStockText();
    }

    /**
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getOutOfStockThreshold()
    {
        return $this->configProvider->outOfStockThreshold();
    }

    /**
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getOutOfStockText()
    {
        return $this->configProvider->outOfStockStockText();
    }

    /**
     * Get the product stock data and methods.
     *
     * @return \Magento\CatalogInventory\Api\StockRegistryInterface
     */
    public function getStockRegistry()
    {
        return $this->stockRegistry;
    }

    /**
     * @return int
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getMaxAllowedQty()
    {
        return $this->configProvider->getMaxAllowedQty();
    }

    /**
     * @param $qty
     * @return \Magento\Framework\Phrase
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getStockStatusMessage($qty)
    {
        $message = $this->getInStockText();
        if ($qty<=$this->getOutOfStockThreshold()) {
            $message = $this->getOutOfStockText();
        } elseif ($qty<=$this->getLowStockThreshold()) {
            $message = $this->getLowStockText();
        }
        return __($message);
    }

    /**
     * @param $product
     * @return string|null
     */
    public function getProductOptions($product)
    {
        $options = null;
        if ($product->getTypeId()=='bundle') {
            //get all the selection products used in bundle product.
            $selectionCollection = $product
                ->getTypeInstance(true)
                ->getSelectionsCollection(
                    $product->getTypeInstance(true)->getOptionsIds($product),
                    $product
                );

            foreach ($selectionCollection as $productSelection) {
                $selectedOptionId = $productSelection->getOptionId();
                $selectedId = $productSelection->getSelectionId();
                $options .= '<input type="hidden" name="bundle_option[' . $selectedOptionId . ']" value="' . $selectedId . '" aria-required="true">';
            }
        }
        return $options;
    }

    /**
     * @param $product
     * @return float|int
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getProductMaxAvailableQty($product)
    {
        $stockRegistry = $this->getStockRegistry();
        $qtyAvailable = 0;
        $max = 0;
        if ($product->getTypeId() === 'simple') {
            $qtyAvailable = $stockRegistry->getStockQty($product->getId());
        } elseif ($product->getTypeId() == 'bundle') {
            $bundleProductId = $product->getId();
            $childrenIds = [];
            $typeInstance = $product->getTypeInstance();
            $requiredChildrenIds = $typeInstance->getChildrenIds($bundleProductId, true);
            if ($requiredChildrenIds) {
                $childrenIds = [];
                foreach ($requiredChildrenIds as $groupedChildrenIds) {
                    $childrenIds = array_merge($childrenIds, $groupedChildrenIds);
                }
            }

            $lowestQuantity = 99999;
            foreach ($childrenIds as $childrenId) {
                $childProductQuantity = $stockRegistry->getStockQty($childrenId);
                if ($lowestQuantity > $childProductQuantity) {
                    $lowestQuantity = $childProductQuantity;
                }
            }
            $qtyAvailable = $lowestQuantity;
        }

        if ($qtyAvailable>0) {
            $max = ($qtyAvailable<=$this->getMaxAllowedQty()) ? $qtyAvailable : $this->getMaxAllowedQty();
        }

        return $max;
    }
}
