<?php
declare(strict_types=1);

namespace BrittainWynyard\OneAsics\Controller\Login;

use Magento\Directory\Model\Config\Source\Country;
use \Magento\Framework\Controller\ResultFactory;
use Overdose\DataLayer\Helper\Cookies;
use Magento\Framework\App\Response\RedirectInterface;
use Magento\Framework\Stdlib\CookieManagerInterface;
use Magento\Framework\Stdlib\Cookie\CookieMetadataFactory;

class Index extends \Magento\Framework\App\Action\Action
{

    protected $resultPageFactory;

    protected $cookiesHelper;

    /**
     * @var Country
     */
    protected $country;

    /**
     * @var RedirectInterface
     */
    protected $redirect;

    protected $cookieManager;
    protected $cookieMetadataFactory;

    protected $countries = [];

    /**
     * Constructor
     *
     * @param \Magento\Framework\App\Action\Context  $context
     * @param \Magento\Framework\View\Result\PageFactory $resultPageFactory
     */
    public function __construct(
        \Magento\Framework\App\Action\Context $context,
        \Magento\Framework\View\Result\PageFactory $resultPageFactory,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Magento\Customer\Model\Session $customerSession,
        \Magento\Checkout\Model\Session $checkoutSession,
        \Magento\Framework\Session\SessionManagerInterface $coreSession,
        \BrittainWynyard\OneAsics\Helper\Config $helper,
        \Magento\Framework\Api\SearchCriteriaBuilder $searchCriteriaBuilder,
        \Magento\Customer\Model\ResourceModel\CustomerRepository $customerRepository,
        \Magento\Customer\Model\CustomerFactory $customerFactory,
        \Psr\Log\LoggerInterface $logger,
        \BrittainWynyard\OneAsics\OAuth2\Client\Provider\OneAsics $oneAsicsProvider,
        \Magento\Newsletter\Model\SubscriberFactory $subscriberFactory,
        Cookies $cookiesHelper,
        Country $country,
        RedirectInterface $redirect,
        CookieManagerInterface $cookieManager,
        CookieMetadataFactory $cookieMetadataFactory
    ) {
        $this->storeManager = $storeManager;
        $this->resultPageFactory = $resultPageFactory;
        $this->customerSession = $customerSession;
        $this->checkoutSession = $checkoutSession;
        $this->configHelper = $helper;
        $this->resultFactory = $context->getResultFactory();
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->customerRepository = $customerRepository;
        $this->customerFactory = $customerFactory;
        $this->logger = $logger;
        $this->oneAsicsProvider = $oneAsicsProvider;
        $this->coreSession = $coreSession;
        $this->subscriberFactory = $subscriberFactory;
        $this->cookiesHelper = $cookiesHelper;
        $this->country = $country;
        $this->redirect = $redirect;
        $this->cookieManager = $cookieManager;
        $this->cookieMetadataFactory = $cookieMetadataFactory;

        parent::__construct($context);
    }

    /**
     * Execute view action
     *
     * @return \Magento\Framework\Controller\ResultInterface
     */
    public function execute()
    {

        //Check if the customer is logged in
        if (!$this->configHelper->isEnabled()) {
            return $this->disabledRedirect();
        }

        //Check if the customer is logged in
        if ($this->customerSession->isLoggedIn()) {
            return $this->loggedInRedirect();
        }

        if($this->_request->getParam('state') && !$this->_request->getParam('code')) {
            $resultRedirect = $this->resultFactory->create(ResultFactory::TYPE_REDIRECT);
            return $resultRedirect->setPath('customer/account/login');
        }

        if (!$this->_request->getParam('state')) {

            $metadata = $this->cookieMetadataFactory->createPublicCookieMetadata()
                                                    ->setDuration(3600)
                                                    ->setPath('/')
                                                    ->setHttpOnly(false);
            $referrerUrl = $this->redirect->getRefererUrl();
            $klaviyoReferrer = $this->getKlaviyoReferrer($referrerUrl);

            $this->cookieManager->setPublicCookie('redirect_link', $klaviyoReferrer, $metadata);

            $this->coreSession->setOneAsicsReferrer('');

            // If we don't have an authorization code then get one
            $authUrl = $this->oneAsicsProvider->getAuthorizationUrl();


            if ($this->configHelper->isDebugMode()) {
                $this->logger->info($authUrl);
            }

            $_SESSION['oauth2state'] = $this->oneAsicsProvider->getState();

            if ($this->_request->getParam('referrer')) {
                $this->coreSession->setOneAsicsReferrer($this->_request->getParam('referrer'));
            }

            return $this->getResponse()->setRedirect(
                $authUrl,
                302
            );
        // Check given state against previously stored one to mitigate CSRF attack
        } else {

            try {
                // Try to get an access token (using the authorization code grant)
                $token = $this->oneAsicsProvider->getAccessToken('authorization_code', [
                    'code' => $this->_request->getParam('code')
                ]);

                // We got an access token, let's now get the user's details
                $user = $this->oneAsicsProvider->getResourceOwner($token);

                if ($this->configHelper->isDebugMode()) {
                    $this->logger->info(print_r($user, true));
                }

                //look for an existing customer with the asics ID
                $customers = $this->findCustomersById($user['asics_id']);


                // if no customer exists then we need to create a new one
                if (count($customers) == 0) {
                    $customers = $this->findCustomersByEmail($user['email']);
                }

                if (count($customers) == 0) {
                    $customer = $this->createNewCustomer($user);
                    $customers = $this->findCustomersById($user['asics_id']);
                    $customer = current($customers);
                } else {
                    $customer = current($customers);
                    $this->updateCustomerDetails($customer, $user);
                }

                return $this->loginCustomer($customer);

            } catch (\Exception $e) {
                $this->messageManager->addWarningMessage("An error occurred logging in ");
                $this->logger->critical(
                    'Error during OneAsics Login',
                    [
                        'params' => $this->_request->getParams(),
                        'exception' => $e,
                    ]
                );
                $resultRedirect = $this->resultFactory->create(ResultFactory::TYPE_REDIRECT);
                return $resultRedirect->setPath('customer/account/login');
            }
        }
    }



    /**
     * redirect logged in customer to account section
     *
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    protected function loggedInRedirect()
    {
        $this->messageManager->addWarningMessage("Already logged in");
        $resultRedirect = $this->resultFactory->create(ResultFactory::TYPE_REDIRECT);
        $resultRedirect->setPath('customer/account');

        return $resultRedirect;
    }


    /**
     * redirect to standard login page if module is disabled
     *
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    protected function disabledRedirect()
    {
        $this->messageManager->addWarningMessage("This login method is disabled");

        $resultRedirect = $this->resultFactory->create(ResultFactory::TYPE_REDIRECT);
        $resultRedirect->setPath('customer/account/login');
        return $resultRedirect;
    }


    /**
     * Find customers by username
     *
     * @param $username
     * @return \Magento\Customer\Api\Data\CustomerInterface[]
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function findCustomersById($id)
    {
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter('oneasics_id', $id)
            ->setPageSize(1)->create();

        $result = $this->customerRepository->getList($searchCriteria);

        return $result->getItems();
    }

    /**
     * Find customers by email
     *
     * @param $email
     * @return \Magento\Customer\Api\Data\CustomerInterface[]
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function findCustomersByEmail($email)
    {
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter('email', $email)
            ->setPageSize(1)->create();

        $result = $this->customerRepository->getList($searchCriteria);

        return $result->getItems();
    }

    protected function getAvailableCountry()
    {
        if (!$this->countries) {
            $countryList = $this->country->toOptionArray();
            foreach ($countryList as $country) {
                $this->countries[] = $country['value'];
            }
        }

        return $this->countries;
    }

    /**
     * Create a new customer in Magento with SSO customer data
     *
     * @param $customerInfo
     * @return \Magento\Customer\Model\Customer
     * @throws \Exception
     */
    protected function createNewCustomer($user)
    {
        $availableCountry = $this->getAvailableCountry();
        $customer = $this->customerFactory->create();
        $customer->setEmail($user['email']);
        $customer->setData('oneasics_id', $user['asics_id']);
        $customer->setFirstname($user['first_name']);
        $customer->setMiddlename("");
        $customer->setLastname($user['last_name']);
        $customer->setData('oneasics_country', $user['country']);
        if (isset($availableCountry[$user['country']])) {
            $customer->setData('location_of_residence', $user['country']);
        }
        $customer->setData('interested_in_gender', $this->configHelper->getInterestedInGenderFromGender($user['gender']));

        $customer->setDob($user["birthday"]);
        $customer->save();

        // if the user has opted to receive a newsletter in one asics, create a subscription in magento
        try {
            if ($user['receive_newsletter']) {
                $this->subscriberFactory->create()->subscribeCustomerById($customer->getId());
            }
        } catch(\Exception $e) {
            $this->logger->error('could not subscribe user '.$user['email'].' to email');
        }


        return $customer;
    }

    protected function updateCustomerDetails($customer, $user)
    {
        $customer->setEmail($user['email']);

        // if for whatever reason the customer doesn't already have a oneasics ID, set it on login
        if ($customer->getCustomAttribute('oneasics_id') == '') {
            $customer->setCustomAttribute('oneasics_id', $user['asics_id']);
        }

        $this->customerRepository->save($customer);
    }

    /**
     * Log customer in and redirect to homepage
     *
     * @param $customer
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    protected function loginCustomer($customer)
    {
        $this->customerSession->loginById($customer->getId());
       // $this->messageManager->addSuccessMessage("Logged in successfully");


        $this->customerSession->setCustomerDataAsLoggedIn($customer);

        //on success page, link the last order
        if ($this->referredBy('success')) {
            $this->linkLastOrder();
        }


        $resultRedirect = $this->resultFactory->create(ResultFactory::TYPE_REDIRECT);
        $resultRedirect->setPath($this->getReturnUrl());
        return $resultRedirect;
    }

    protected function linkLastOrder()
    {
        $customer = $this->customerSession->getCustomer();

        if (!$customer->getId()) {
            return;
        }

        $lastOrder = $this->checkoutSession->getLastRealOrder();

        if (!$lastOrder->getId() || $lastOrder->getCustomerId()) {
            return;
        }

        $lastOrder->setCustomerId($customer->getId());
        $lastOrder->save();
    }

    protected function getReturnUrl()
    {
        $customerEmail = '';
        $event = 'account_login';
        $identification_location = 'login_site';
        $customer = $this->customerSession->getCustomer();

        if ($customer->getEmail()) {
            $customerEmail = strtolower($customer->getEmail());
        }


        if ($this->referredBy('checkout')) {
            $identification_location = 'login_checkout';
            $returnUrl = 'checkout?login-success=true';
        } else {
            $returnUrl = 'customer/account?login-success=true';
        }

        $this->cookiesHelper->setDataLayerCookieUserIdentify($customerEmail);
        $this->cookiesHelper->setDataLayerCookie($event, $identification_location);

        return $returnUrl;
    }

    /**
     * @param $name
     * @return bool
     */
    protected function referredBy($name)
    {
        return ($this->coreSession->getOneAsicsReferrer() && $this->coreSession->getOneAsicsReferrer() == $name);
    }

    /**
     * @param $url
     * @return string
     */
    public function getKlaviyoReferrer($url)
    {
        try {
            $path = parse_url($url, PHP_URL_PATH);
            if (empty($path)) {
                return 'direct';
            }
            $pattern = '~^/[a-z]{2}/[a-z]{2}-[a-z]{2}~i';
            $stripPath = preg_replace($pattern, '', $path);
            return trim($stripPath, '/');
        } catch (\Exception $e) {
            return $url;
        }
    }
}

